package com.suncode.pwfl.assistant.agent.variable.tools;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.suncode.pwfl.assistant.AgentType;
import com.suncode.pwfl.assistant.tools.Tool;
import com.suncode.pwfl.assistant.tools.ToolContext;
import com.suncode.pwfl.assistant.tools.ToolDefinition;
import com.suncode.pwfl.assistant.tools.ToolResult;
import com.suncode.pwfl.workflow.process.map.Activity;
import com.suncode.pwfl.workflow.process.map.Process;
import com.suncode.pwfl.workflow.process.map.VariableRef;
import lombok.Getter;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * Tool for retrieving all variables assigned to specific activities.
 */
@Getter
public class GetVariablesAssignedToActivitiesTool implements Tool
{
    public static final ToolDefinition DEFINITION = new ToolDefinition(
        AgentType.VARIABLE,
        "get_variables_assigned_to_activities",
        "Returns all variables assigned to one or more activities. Returns a list of activities with their assigned variables in the following format: { \"activities\": [ { \"activityId\": \"<id>\", \"assignedVariables\": [ { \"variableId\": \"<id>\", \"visibilityType\": \"<type>\" } ] } ] }",
        "To get all variables assigned to activities write { \"activityIds\": [\"<id1>\", \"<id2>\"] }.",
        GetVariablesAssignedToActivitiesTool.class
    );

    private final List<String> activityIds;

    @JsonCreator
    public GetVariablesAssignedToActivitiesTool(
        @JsonProperty("activityIds") List<String> activityIds)
    {
        this.activityIds = activityIds;
    }

    @Override
    public ToolResult execute(ToolContext toolContext )
    {
        if (activityIds == null || activityIds.isEmpty())
        {
            return new ToolResult(
                false,
                "Error: Invalid payload. Expecting an object with a non-empty \"activityIds\" array of strings."
            );
        }

        Process process = toolContext.getProcess();
        List<String> results = new ArrayList<>();
        List<String> errors = new ArrayList<>();

        for (String activityId : activityIds)
        {
            Optional<Activity> activityOpt = process.getActivities().stream()
                .filter(activity -> activity.getId().equals(activityId))
                .findFirst();

            if (activityOpt.isEmpty())
            {
                errors.add("Error: Activity with activityId: '" + activityId + "' doesn't exist!.");
                continue;
            }

            Activity activity = activityOpt.get();
            if (activity.getVariableRefs().isEmpty())
            {
                results.add("Activity '" + activityId + "': No variables assigned yet.");
                continue;
            }

            String formattedVariables = "";
            int index = 1;
            for (VariableRef variableRef : activity.getVariableRefs())
            {
                if (!formattedVariables.isEmpty())
                {
                    formattedVariables += "\n";
                }
                formattedVariables += "  " + index + ". variableId: " + variableRef.getId() + 
                                    ", visibilityType: " + variableRef.getType();
                index++;
            }
            results.add("Activity '" + activityId + "':\n" + formattedVariables);
        }

        String data = "";
        if (results.size() > 0)
        {
            data += "Successfully retrieved assigned variables:\n" + String.join("\n", results);
        }

        if (errors.size() > 0)
        {
            if (!data.isEmpty())
            {
                data += "\n\n";
            }
            data += "Errors:\n" + String.join("\n", errors);
        }

        if (results.isEmpty() && errors.size() > 0)
        {
            return new ToolResult(false, data);
        }

        return new ToolResult(true, data.isEmpty() ? "No activities found or processed." : data);
    }
}
