package com.suncode.pwfl.assistant.agent.variable.tools;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.suncode.pwfl.assistant.AgentType;
import com.suncode.pwfl.assistant.tools.Tool;
import com.suncode.pwfl.assistant.tools.ToolContext;
import com.suncode.pwfl.assistant.tools.ToolDefinition;
import com.suncode.pwfl.assistant.tools.ToolResult;
import com.suncode.pwfl.workflow.process.map.Activity;
import com.suncode.pwfl.workflow.process.map.Process;
import lombok.Getter;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Tool for deleting variables from the process map.
 * Also removes variable assignments from all activities.
 */
@Getter
public class DeleteVariableTool implements Tool
{
    public static final ToolDefinition DEFINITION = new ToolDefinition(
        AgentType.VARIABLE,
        "delete_variable",
        "Deletes one or more variables from the process map. Returns the deleted variable IDs in the following format: { \"deletedVariableIds\": [\"<id1>\", \"<id2>\"] }",
        "To delete variables write { \"variableIds\": [\"<id1>\", \"<id2>\"] }.",
        DeleteVariableTool.class
    );

    private final List<String> variableIds;

    @JsonCreator
    public DeleteVariableTool(@JsonProperty("variableIds") List<String> variableIds)
    {
        this.variableIds = variableIds;
    }

    @Override
    public ToolResult execute(ToolContext toolContext )
    {
        if (variableIds == null || variableIds.isEmpty())
        {
            return new ToolResult(
                false,
                "Error: Invalid payload. Expecting an object with a non-empty \"variableIds\" array."
            );
        }

        Process process = toolContext.getProcess();
        List<String> deletedIds = new ArrayList<>();
        List<String> notFoundIds = new ArrayList<>();
        Set<String> idsToDelete = new HashSet<>(variableIds);
        Set<String> currentVariableIds = new HashSet<>();
        
        process.getVariables().forEach(v -> currentVariableIds.add(v.getId()));

        for (String id : idsToDelete)
        {
            if (currentVariableIds.contains(id))
            {
                deletedIds.add(id);
            }
            else
            {
                notFoundIds.add(id);
            }
        }

        if (deletedIds.size() > 0)
        {
            // Filter out the variables to be deleted
            process.getVariables().removeIf(variable -> idsToDelete.contains(variable.getId()));

            // TODO: Clean up variable references from gateway transition conditions

            // Remove assignments of this variable from all activities
            for (Activity activity : process.getActivities())
            {
                activity.getVariableRefs().removeIf(variableRef -> 
                    idsToDelete.contains(variableRef.getId()));
            }
        }

        String data = "";
        if (deletedIds.size() > 0)
        {
            data += "Successfully deleted variables with variableId's: " + 
                   String.join(", ", deletedIds) + ".";
        }

        if (notFoundIds.size() > 0)
        {
            if (!data.isEmpty())
            {
                data += "\n";
            }
            data += "Variables with the following variableId's were not found: " + 
                   String.join(", ", notFoundIds) + ".";
        }

        if (deletedIds.isEmpty() && !notFoundIds.isEmpty())
        {
            return new ToolResult(
                false,
                "No variables deleted. Variables with the following variableId's were not found: " + 
                String.join(", ", notFoundIds) + "."
            );
        }

        return new ToolResult(true, data);
    }
}
