package com.suncode.pwfl.assistant.agent.variable.tools;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.suncode.pwfl.assistant.AgentType;
import com.suncode.pwfl.assistant.tools.Tool;
import com.suncode.pwfl.assistant.tools.ToolContext;
import com.suncode.pwfl.assistant.tools.ToolDefinition;
import com.suncode.pwfl.assistant.tools.ToolResult;
import com.suncode.pwfl.workflow.process.map.Process;
import lombok.Getter;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Tool for deleting activities from the process map.
 * Also cleans up associated transitions.
 */
@Getter
public class DeleteActivityTool implements Tool
{
    public static final ToolDefinition DEFINITION = new ToolDefinition(
        AgentType.VARIABLE,
        "delete_activity",
        "Deletes one or more activities from the process map. Returns the deleted activity IDs in the following format: { \"deletedActivityIds\": [\"<id1>\", \"<id2>\"] }",
        "To delete activities write { \"activityIds\": [\"<id1>\", \"<id2>\"] }.",
        DeleteActivityTool.class
    );

    private final List<String> activityIds;

    @JsonCreator
    public DeleteActivityTool(@JsonProperty("activityIds") List<String> activityIds)
    {
        this.activityIds = activityIds;
    }

    @Override
    public ToolResult execute(ToolContext toolContext )
    {
        if (activityIds == null || activityIds.isEmpty())
        {
            return new ToolResult(
                false,
                "Error: Invalid payload. Expecting an object with a non-empty \"activityIds\" array of strings."
            );
        }

        Process process = toolContext.getProcess();
        List<String> deletedIds = new ArrayList<>();
        List<String> notFoundIds = new ArrayList<>();
        Set<String> idsToDelete = new HashSet<>(activityIds);
        Set<String> currentActivityIds = new HashSet<>();
        
        process.getActivities().forEach(a -> currentActivityIds.add(a.getId()));

        for (String id : idsToDelete)
        {
            if (currentActivityIds.contains(id))
            {
                deletedIds.add(id);
            }
            else
            {
                notFoundIds.add(id);
            }
        }

        if (deletedIds.size() > 0)
        {
            // Filter out the activities to be deleted
            process.getActivities().removeIf(activity -> idsToDelete.contains(activity.getId()));

            // Clean up transitions
            process.getTransitions().removeIf(transition ->
                idsToDelete.contains(transition.getSourceId()) || 
                idsToDelete.contains(transition.getTargetId()));

            // TODO: Clean up gateway transitions that reference deleted activities
        }

        String data = "";
        if (deletedIds.size() > 0)
        {
            data += "Successfully deleted activities with activityId's: " + String.join(", ", deletedIds) + ".";
        }

        if (notFoundIds.size() > 0)
        {
            if (!data.isEmpty())
            {
                data += "\n";
            }
            data += "Activities with the following activityId's were not found: " + 
                    String.join(", ", notFoundIds) + ".";
        }

        if (deletedIds.isEmpty() && !notFoundIds.isEmpty())
        {
            return new ToolResult(
                false,
                "No activities deleted. Activities with the following activityId's were not found: " + 
                String.join(", ", notFoundIds) + "."
            );
        }

        return new ToolResult(true, data);
    }
}
