package com.suncode.pwfl.assistant.agent.variable.tools;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.suncode.pwfl.assistant.AgentType;
import com.suncode.pwfl.assistant.tools.Tool;
import com.suncode.pwfl.assistant.tools.ToolContext;
import com.suncode.pwfl.assistant.tools.ToolDefinition;
import com.suncode.pwfl.assistant.tools.ToolResult;
import com.suncode.pwfl.util.IdNormalizer;
import com.suncode.pwfl.workflow.process.map.Process;
import com.suncode.pwfl.workflow.process.map.Variable;
import com.suncode.pwfl.workflow.process.map.VariableType;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;

import java.util.ArrayList;
import java.util.List;

/**
 * Tool for creating new variables in the business process map.
 * A variable represents a field on an activity form.
 * VariableId is generated automatically based on variableName.
 */
@Getter
public class CreateVariablesTool implements Tool
{
    public static final ToolDefinition DEFINITION = new ToolDefinition(
        AgentType.VARIABLE,
        "create_variables",
        """
            Creates one or more new variables in the business process map.
            A variable is a component that represents a field on a activity form in which values of a specific type can be placed.
            VariableId is generated automatically based on each variableName.
            """,
        "To create variables, write { \"variables\": [{ \"variableName\": \"<name1>\", \"variableType\": \"<type1>\" }, { \"variableName\": \"<name2>\", \"variableType\": \"<type2>\" }] }. Payload of the variable names MUST be in the nominative case, for example: Invoice number, Invoice date, Invoice amount. Allowed variable types: 'STRING', 'INTEGER', 'FLOAT', 'BOOLEAN', 'DATE', 'DATE_TIME'",
        CreateVariablesTool.class
    );

    private final List<VariableToCreate> variables;

    @JsonCreator
    public CreateVariablesTool(@JsonProperty("variables") List<VariableToCreate> variables)
    {
        this.variables = variables;
    }

    @Override
    public ToolResult execute(ToolContext toolContext )
    {
        if (variables == null || variables.isEmpty())
        {
            return new ToolResult(
                false,
                "Error: Invalid payload. Expecting an object with a non-empty \"variables\" array."
            );
        }

        Process process = toolContext.getProcess();
        List<CreatedVariable> createdVariables = new ArrayList<>();
        List<String> errors = new ArrayList<>();

        for (VariableToCreate variableToCreate : variables)
        {
            String baseVariableId = IdNormalizer.normalize(variableToCreate.getVariableName());
            String variableId = generateUniqueVariableId(process, baseVariableId);

            Variable variable = new Variable(
                variableToCreate.getVariableType(),
                variableId,
                variableToCreate.getVariableName()
            );
            process.getVariables().add(variable);

            createdVariables.add(new CreatedVariable(
                variableId,
                variableToCreate.getVariableName(),
                variableToCreate.getVariableType().name()
            ));
        }

        String responseData = "";
        if (createdVariables.size() > 0)
        {
            String formattedCreated = createdVariables.stream()
                .map(v -> "variableId: " + v.variableId() + 
                         ", variableName: '" + v.variableName() + 
                         "', type: " + v.variableType())
                .reduce("", (acc, item) -> acc.isEmpty() ? item : acc + "\n" + item);
            responseData += "Successfully created " + createdVariables.size() + 
                          " variables:\n" + formattedCreated;
        }

        if (errors.size() > 0)
        {
            if (!responseData.isEmpty())
            {
                responseData += "\n\n";
            }
            responseData += "Encountered errors:\n" + String.join("\n", errors);
        }

        return new ToolResult(createdVariables.size() > 0, responseData);
    }

    /**
     * Generates a unique variable ID based on the base ID
     */
    private String generateUniqueVariableId(Process process, String baseVariableId)
    {
        String variableId = baseVariableId;
        int counter = 1;
        final String baseId = baseVariableId;

        while (true)
        {
            final String currentId = variableId;
            boolean exists = process.getVariables().stream()
                .anyMatch(v -> v.getId().equals(currentId));
            
            if (!exists)
            {
                break;
            }
            
            String counterStr = String.valueOf(counter);
            int maxLength = 16 - counterStr.length();
            String trimmedId = baseId.length() > maxLength ? 
                              baseId.substring(0, maxLength) : baseId;
            variableId = trimmedId + counter;
            counter++;
        }

        return variableId;
    }

    private record CreatedVariable(String variableId, String variableName, String variableType)
    {
    }

    @Getter
    @Setter
    @AllArgsConstructor
    @NoArgsConstructor
    public static class VariableToCreate
    {
        @JsonProperty(required = true)
        private String variableName;

        @JsonProperty(required = true)
        private VariableType variableType;
    }
}
