package com.suncode.pwfl.assistant.agent.variable.tools;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.suncode.pwfl.assistant.AgentType;
import com.suncode.pwfl.assistant.tools.Tool;
import com.suncode.pwfl.assistant.tools.ToolContext;
import com.suncode.pwfl.assistant.tools.ToolDefinition;
import com.suncode.pwfl.assistant.tools.ToolResult;
import com.suncode.pwfl.workflow.process.map.Activity;
import com.suncode.pwfl.workflow.process.map.Process;
import lombok.Getter;

import java.text.Normalizer;
import java.util.ArrayList;
import java.util.List;

/**
 * Tool for creating new activities in the process map.
 * ActivityId is generated automatically based on each activityName.
 */
@Getter
public class CreateActivitiesTool implements Tool
{
    public static final ToolDefinition DEFINITION = new ToolDefinition(
        AgentType.VARIABLE,
        "create_activities",
        "Creates one or more new activities in the process map. ActivityId is generated automatically based on each activityName. Returns the created activities in the following format: { \"createdActivities\": [ { \"activityId\": \"<id>\", \"activityName\": \"<name>\" } ] }",
        "To create activities write { \"activityNames\": [\"<name of activity 1>\", \"<name of activity 2>\"] }. Payload of the activity names MUST be in the nominative case, for example: Invoice registration, Content description, Accounting.",
        CreateActivitiesTool.class
    );

    private final List<String> activityNames;

    @JsonCreator
    public CreateActivitiesTool(@JsonProperty("activityNames") List<String> activityNames)
    {
        this.activityNames = activityNames;
    }

    @Override
    public ToolResult execute(ToolContext toolContext )
    {
        if (activityNames == null || activityNames.isEmpty())
        {
            return new ToolResult(
                false, 
                "Error: Invalid payload. Expecting a non-empty \"activityNames\" array of strings."
            );
        }

        Process process = toolContext.getProcess();
        List<CreatedActivity> createdActivities = new ArrayList<>();

        for (String activityName : activityNames)
        {
            String activityId = generateUniqueActivityId(process, activityName);
            
            Activity activity = new Activity(activityId, activityName);
            process.getActivities().add(activity);

            createdActivities.add(new CreatedActivity(activity.getId(), activity.getName()));
        }

        String formattedData = createdActivities.stream()
            .map(activity -> "activityId: " + activity.activityId() + ", activityName: " + activity.activityName())
            .reduce("", (acc, item) -> acc.isEmpty() ? item : acc + "\n" + item);

        return new ToolResult(
            true, 
            "Successfully created " + createdActivities.size() + " activities:\n" + formattedData
        );
    }

    /**
     * Generates a unique activity ID based on the activity name
     */
    private String generateUniqueActivityId(Process process, String activityName)
    {
        // Generate unique ID matching TypeScript logic
        String baseActivityId = Normalizer.normalize(activityName, Normalizer.Form.NFD)
            .toLowerCase()
            .replaceAll("[\\u0300-\\u036f]", "")
            .replaceAll("\\s+", "_")
            .replaceAll("[^a-z0-9_]", "");
        
        if (baseActivityId.length() > 16)
        {
            baseActivityId = baseActivityId.substring(0, 16);
        }

        final String baseId = baseActivityId;
        String activityId = baseId;
        int counter = 1;

        while (true)
        {
            final String currentId = activityId;
            boolean exists = process.getActivities().stream()
                .anyMatch(a -> a.getId().equals(currentId));
            
            if (!exists)
            {
                break;
            }
            
            String counterStr = String.valueOf(counter);
            int maxLength = 16 - counterStr.length();
            String trimmedId = baseId.length() > maxLength ? baseId.substring(0, maxLength) : baseId;
            activityId = trimmedId + counter;
            counter++;
        }

        return activityId;
    }

    public record CreatedActivity(String activityId, String activityName)
    {
    }
}
