package com.suncode.pwfl.assistant.agent.variable.tools;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.suncode.pwfl.assistant.AgentType;
import com.suncode.pwfl.assistant.tools.Tool;
import com.suncode.pwfl.assistant.tools.ToolContext;
import com.suncode.pwfl.assistant.tools.ToolDefinition;
import com.suncode.pwfl.assistant.tools.ToolResult;
import com.suncode.pwfl.workflow.process.map.Process;
import com.suncode.pwfl.workflow.process.map.Variable;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.lucene.analysis.standard.StandardAnalyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.TextField;
import org.apache.lucene.index.DirectoryReader;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.FuzzyQuery;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TopDocs;
import org.apache.lucene.store.ByteBuffersDirectory;
import org.apache.lucene.store.Directory;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

/**
 * Tool for finding variables by name using fuzzy search.
 * Variables are components representing fields on activity forms
 * for storing data of specific types.
 */
@Slf4j
@Getter
public class CheckVariablesExistByVariableNameTool implements Tool
{
    public static final ToolDefinition DEFINITION = new ToolDefinition(
        AgentType.VARIABLE,
        "check_variables_exist_by_variableName",
        """
            Use this tool to search for process map variable(s).
            Tool finds variables by providing a list of their names using fuzzy search.
            Variables are used for storing data. Variable is a component that represents a field on a activity form in which values a specific type can be placed.
            (VARIABLES DO NOT STORE ACCEPTANCE BUTTONS).
            Returns the variables found by variableName in the following format: { "variableId": "<id>", "variableName": "<name>", "variableType": "<type>" }.
            Variables are used for storing data on a form.
            """,
        "To find variables write { \"variableNames\": [\"<name of variable 1>\", \"name of variable 2\"] }.",
        CheckVariablesExistByVariableNameTool.class
    );

    private final List<String> variableNames;

    @JsonCreator
    public CheckVariablesExistByVariableNameTool(
        @JsonProperty("variableNames") List<String> variableNames)
    {
        this.variableNames = variableNames;
    }

    @Override
    public ToolResult execute(ToolContext toolContext )
    {
        if (variableNames == null || variableNames.isEmpty())
        {
            return new ToolResult(
                false,
                "Error: Invalid payload. Expecting an object with a non-empty \"variableNames\" array of strings."
            );
        }

        try
        {
            Process process = toolContext.getProcess();
            Directory dir = new ByteBuffersDirectory();
            StandardAnalyzer analyzer = new StandardAnalyzer();

            IndexWriterConfig config = new IndexWriterConfig(analyzer);
            IndexWriter writer = new IndexWriter(dir, config);

            for (Variable variable : process.getVariables())
            {
                addDoc(writer, variable.getId(), variable.getName());
            }

            writer.close();

            IndexReader reader = DirectoryReader.open(dir);
            IndexSearcher searcher = new IndexSearcher(reader);
            
            List<FoundVariable> foundVariables = new ArrayList<>();
            Set<String> foundVariableIds = new HashSet<>();
            List<String> notFoundNames = new ArrayList<>();

            for (String name : variableNames)
            {
                Term term = new Term("variableName", name);
                Query query = new FuzzyQuery(term, 2);

                TopDocs hits = searcher.search(query, 10);

                if (hits.scoreDocs.length > 0)
                {
                    Document d = searcher.storedFields().document(hits.scoreDocs[0].doc);
                    String variableId = d.get("variableId");
                    
                    Optional<Variable> variableOpt = process.getVariables().stream()
                        .filter(var -> var.getId().equals(variableId))
                        .findFirst();

                    if (variableOpt.isPresent() && !foundVariableIds.contains(variableId))
                    {
                        Variable variable = variableOpt.get();
                        foundVariables.add(new FoundVariable(
                            variable.getId(),
                            variable.getName(),
                            variable.getType().name()
                        ));
                        foundVariableIds.add(variableId);
                    }
                }
                else
                {
                    notFoundNames.add(name);
                }
            }

            reader.close();
            dir.close();

            String data = "";
            if (foundVariables.size() > 0)
            {
                String formattedFound = foundVariables.stream()
                    .map(v -> "variableId: " + v.variableId() + 
                             ", variableName: '" + v.variableName() + 
                             "', variableType: " + v.variableType())
                    .reduce("", (acc, item) -> acc.isEmpty() ? item : acc + "\n" + item);
                data += "Found variables:\n" + formattedFound;
            }

            if (notFoundNames.size() > 0)
            {
                if (!data.isEmpty())
                {
                    data += "\n";
                }
                data += "Variables not found for names: " + String.join(", ", notFoundNames) + 
                       ". Probably they doesn't exsist in the process map.";
            }

            if (foundVariables.isEmpty())
            {
                return new ToolResult(
                    true,
                    "No variables found for the given names: " + String.join(", ", notFoundNames) + 
                    ". Probably they doesn't exsist in the process map."
                );
            }

            return new ToolResult(true, data);
        }
        catch (Exception e)
        {
            log.error(e.getMessage(), e);
            return new ToolResult(false, "Error: " + e.getMessage());
        }
    }

    private record FoundVariable(String variableId, String variableName, String variableType)
    {
    }

    private void addDoc(IndexWriter w, String variableId, String variableName) throws Exception
    {
        Document doc = new Document();
        doc.add(new TextField("variableId", variableId, Field.Store.YES));
        doc.add(new TextField("variableName", variableName, Field.Store.YES));
        w.addDocument(doc);
    }
}
