package com.suncode.pwfl.assistant.agent.variable.tools;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.suncode.pwfl.assistant.AgentType;
import com.suncode.pwfl.assistant.tools.Tool;
import com.suncode.pwfl.assistant.tools.ToolContext;
import com.suncode.pwfl.assistant.tools.ToolDefinition;
import com.suncode.pwfl.assistant.tools.ToolResult;
import com.suncode.pwfl.workflow.process.map.Activity;
import com.suncode.pwfl.workflow.process.map.Process;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.lucene.analysis.standard.StandardAnalyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.TextField;
import org.apache.lucene.index.DirectoryReader;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.FuzzyQuery;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TopDocs;
import org.apache.lucene.store.ByteBuffersDirectory;
import org.apache.lucene.store.Directory;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Tool for finding activities by name using fuzzy search.
 * Searches for activities by providing a list of their names.
 */
@Slf4j
@Getter
public class CheckActivitiesExistByActivityNameTool implements Tool
{
    public static final ToolDefinition DEFINITION = new ToolDefinition(
        AgentType.VARIABLE,
        "check_activities_exist_by_activityName",
        "Search for activity(ies) details and finds activities by providing a list of their names using fuzzy search. Returns the activities found by activityName in the following format: { \"activityId\": \"<id>\", \"activityName\": \"<name>\" }",
        "To find activities write { \"activityNames\": [\"<name of activity 1>\", \"name of activity 2\"] }.",
        CheckActivitiesExistByActivityNameTool.class
    );

    private final List<String> activityNames;

    @JsonCreator
    public CheckActivitiesExistByActivityNameTool(
        @JsonProperty("activityNames") List<String> activityNames)
    {
        this.activityNames = activityNames;
    }

    @Override
    public ToolResult execute(ToolContext toolContext )
    {
        if (activityNames == null || activityNames.isEmpty())
        {
            return new ToolResult(
                false,
                "Error: Invalid payload. Expecting an object with a non-empty \"activityNames\" array of strings."
            );
        }

        try
        {
            Process process = toolContext.getProcess();
            Directory dir = new ByteBuffersDirectory();
            StandardAnalyzer analyzer = new StandardAnalyzer();

            IndexWriterConfig config = new IndexWriterConfig(analyzer);
            IndexWriter writer = new IndexWriter(dir, config);

            for (Activity activity : process.getActivities())
            {
                addDoc(writer, activity.getId(), activity.getName());
            }

            writer.close();

            IndexReader reader = DirectoryReader.open(dir);
            IndexSearcher searcher = new IndexSearcher(reader);
            
            List<FoundActivity> foundActivities = new ArrayList<>();
            Set<String> foundActivityIds = new HashSet<>();
            List<String> notFoundNames = new ArrayList<>();

            for (String name : activityNames)
            {
                Term term = new Term("activityName", name);
                Query query = new FuzzyQuery(term, 2);

                TopDocs hits = searcher.search(query, 10);

                if (hits.scoreDocs.length > 0)
                {
                    Document d = searcher.storedFields().document(hits.scoreDocs[0].doc);
                    String activityId = d.get("activityId");
                    String activityName = d.get("activityName");
                    
                    if (!foundActivityIds.contains(activityId))
                    {
                        foundActivities.add(new FoundActivity(activityId, activityName));
                        foundActivityIds.add(activityId);
                    }
                }
                else
                {
                    notFoundNames.add(name);
                }
            }

            reader.close();
            dir.close();

            String data = "";
            if (foundActivities.size() > 0)
            {
                String formattedFound = foundActivities.stream()
                    .map(a -> "activityId: " + a.activityId() + 
                             ", activityName: '" + a.activityName() + "'")
                    .reduce("", (acc, item) -> acc.isEmpty() ? item : acc + "\n" + item);
                data += "Found activities:\n" + formattedFound;
            }

            if (notFoundNames.size() > 0)
            {
                if (!data.isEmpty())
                {
                    data += "\n";
                }
                data += "Activities not found for names: " + String.join(", ", notFoundNames) + 
                       ". Probably they doesn't exsist in the process map.";
            }

            if (foundActivities.isEmpty())
            {
                return new ToolResult(
                    true,
                    "No activities found for the given names: " + String.join(", ", notFoundNames) + 
                    ". Probably they doesn't exsist in the process map."
                );
            }

            return new ToolResult(true, data);
        }
        catch (Exception e)
        {
            log.error(e.getMessage(), e);
            return new ToolResult(false, "Error: " + e.getMessage());
        }
    }

    public record FoundActivity(String activityId, String activityName)
    {
    }

    private void addDoc(IndexWriter w, String activityId, String activityName) throws Exception
    {
        Document doc = new Document();
        doc.add(new TextField("activityId", activityId, Field.Store.YES));
        doc.add(new TextField("activityName", activityName, Field.Store.YES));
        w.addDocument(doc);
    }
}
