package com.suncode.pwfl.assistant.agent.variable.prompts;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.suncode.config.AIAssistantObjectMapper;
import com.suncode.pwfl.assistant.model.Action;
import com.suncode.pwfl.assistant.model.ActionResult;
import com.suncode.pwfl.assistant.model.AgentState;
import com.suncode.pwfl.assistant.model.Task;
import com.suncode.pwfl.assistant.prompting.PromptProvider;
import com.suncode.pwfl.assistant.tools.ToolDefinition;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

@RequiredArgsConstructor
public class VariableActionUsePromptProvider
    implements PromptProvider
{

    private final List<ToolDefinition> tools;

    @SneakyThrows
    public String provide( AgentState state )
    {
        ObjectMapper objectMapper = new AIAssistantObjectMapper();

        String prompt = """
            You're AI agent, preparing to execute an action using a specific tool. Your task is to generate the appropriate payload based on the tool's instruction format.
            
            <prompt_objective>
            Generate a valid payload for the selected tool following its specific instruction format, considering the current context, action details, current task and the results of previously executed tasks.
            </prompt_objective>
            
            <prompt_rules>
            - ALWAYS output a valid JSON object with "_thinking" and "result" properties
            - The "_thinking" property MUST contain your reasoning about payload generation
            - The "result" property MUST be an object matching the tool's instruction format
            - STRICTLY follow the tool's instruction format
            - ANALYZE the results of previously completed tasks provided in the '<all_tasks>' section to gather necessary information for the current payload.
            - Pay attention to the current task provided in the '<current_task>' section to gather necessary information for the current payload.
            - Consider the current context when generating the payload
            - Ensure the payload is relevant to the action's objective
            - ALWAYS Generate payload in the same language as the user's message
            - FORBIDDEN: Generating payloads that don't match the tool's instruction format
            </prompt_rules>
            
            <prompt_examples>
            USER: Create an invoice process for me.
            AI: {
              "_thinking": "The current action is to set the process name. The description for this action is 'Set the process name to Invoice Process'. I need to create a JSON payload with the 'processName' key, as specified in the tool's instructions.",
              "result": {
                "processName": "Invoice Process"
              }
            }
            
            USER: Create a pizza making process with three tasks: kneading dough, adding ingredients, and baking.
            AI: {
              "_thinking": "The current action is to create an activity. The action's description is 'Create the Kneading dough activity with ID kneading_dough'. Based on the tool's instruction, I need to provide 'activityId' and 'activityName'. I will extract these from the description.",
              "result": {
                "activityId": "kneading_dough",
                "activityName": "Kneading dough"
              }
            }
            
            USER: Create a pizza making process with three tasks: kneading dough, adding ingredients, and baking.
            AI: {
              "_thinking": "The current action is to provide the final answer to the user. The description is 'Inform the user that the Pizza Making Process has been fully modeled'. I will use this text for the 'answer' field in the payload.",
              "result": {
                "answer": "The 'Pizza Making Process' has been fully modeled."
              }
            }
            
            </prompt_examples>
            
            <dynamic_context>
            <all_tasks>
            %s
            </all_tasks>
            
            <current_task>
            %s
            </current_task>
            
            <selected_tool>
            %s
            </selected_tool>
            
            <current_action>
            %s
            </current_action>
            </dynamic_context>
            
            <execution_validation>
            Before delivering ANY output:
            - Verify the payload matches the tool's instruction format
            - Verify that your output is in the same language as the user's message
            - Confirm the payload is relevant to the action's objective
            - Validate contextual appropriateness
            </execution_validation>
            """;

        Optional<Task> currentTask = state.getTasks().stream()
            .filter( task -> task.getUuid().equals( state.getTask() ) )
            .findFirst();
        String currentTaskInfo = getCurrentTask( currentTask );
        return String.format( prompt, buildAllTasks( state.getTasks() ), currentTaskInfo, getSelectedTool( currentTask, state ),
                              getCurrentAction( currentTask, state, objectMapper ) );
    }

    private String getCurrentTask( Optional<Task> currentTask )
    {
        StringBuilder currentTaskStringBuilder = new StringBuilder();
        if ( currentTask.isPresent() )
        {
            Task task = currentTask.get();
            currentTaskStringBuilder.append( String.format(
                "<task uuid=\"%s\" name=\"%s\" status=\"%s\">",
                task.getUuid(), task.getName(), task.getState().name()
            ) );
            currentTaskStringBuilder.append( String.format(
                "  <description>%s</description>", task.getDescription()
            ) );
        }
        return currentTaskStringBuilder.toString();
    }

    private String getSelectedTool( Optional<Task> currentTask, AgentState state )
    {
        if ( currentTask.isEmpty() )
        {
            return "";
        }
        Optional<Action> currentAction = currentTask.get().getActions().stream()
            .filter( action -> action.getUuid().equals( state.getAction() ) )
            .findFirst();
        if ( currentAction.isEmpty() )
        {
            return "";
        }

        Optional<String> currentTool = tools.stream()
            .filter( tool -> tool.name().equals( currentAction.get().getToolName() ) )
            .map( ToolDefinition::instruction )
            .findFirst();
        return currentTool.orElse( "" );
    }

    @SneakyThrows
    private String getCurrentAction( Optional<Task> currentTask, AgentState state, ObjectMapper objectMapper )
    {
        if ( currentTask.isEmpty() )
        {
            return "";
        }

        Optional<Action> currentAction = currentTask.get().getActions().stream()
            .filter( action -> action.getUuid().equals( state.getAction() ) )
            .findFirst();
        if ( currentAction.isEmpty() )
        {
            return "";
        }

        return objectMapper.writeValueAsString( currentAction.get() );
    }

    private String buildAllTasks( List<Task> tasks )
    {
        return tasks.stream()
            .map( this::buildTaskTag )
            .collect( Collectors.joining( "\n" ) );
    }

    private String buildTaskTag( Task task )
    {
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append( String.format(
            "<task uuid=\"%s\" name=\"%s\" status=\"%s\">",
            task.getUuid(), task.getName(), task.getState().name()
        ) );
        stringBuilder.append( String.format(
            "  <description>%s</description>", task.getDescription()
        ) );
        stringBuilder.append( "  <result>" );
        String resultData = task.getActions()
            .stream()
            .map( Action::getResult )
            .filter( Objects::nonNull )
            .map( ActionResult::getData )
            .filter( Objects::nonNull )
            .map( Object::toString )
            .collect( Collectors.joining( "\n" ) );
        stringBuilder.append( resultData );
        stringBuilder.append( "  </result>" );
        return stringBuilder.toString();
    }

}
