package com.suncode.pwfl.assistant.agent.transition.tools;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.suncode.pwfl.assistant.AgentType;
import com.suncode.pwfl.assistant.tools.Tool;
import com.suncode.pwfl.assistant.tools.ToolContext;
import com.suncode.pwfl.assistant.tools.ToolDefinition;
import com.suncode.pwfl.assistant.tools.ToolResult;
import com.suncode.pwfl.workflow.process.map.AcceptButton;
import com.suncode.pwfl.workflow.process.map.Activity;
import com.suncode.pwfl.workflow.process.map.Process;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;

import java.text.Normalizer;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * Tool for updating accept buttons on an activity.
 * Generates new buttonId based on new name.
 */
@Getter
public class UpdateAcceptButtonsOnActivityTool implements Tool
{
    public static final ToolDefinition DEFINITION = new ToolDefinition(
        AgentType.TRANSITION,
        "update_accept_buttons_on_activity",
        """
            Updates one or more existing accept buttons on a specified activity.
            A new buttonId is generated for each button based on its new name.
            """,
        "To update accept buttons, write { \"activityId\": \"<id>\", \"updates\": [ { \"acceptButtonId\": \"<id to update>\", \"newAcceptButtonName\": \"<new name>\" } ] }",
        UpdateAcceptButtonsOnActivityTool.class
    );

    private final String activityId;
    private final List<ButtonUpdate> updates;

    @JsonCreator
    public UpdateAcceptButtonsOnActivityTool(
        @JsonProperty("activityId") String activityId,
        @JsonProperty("updates") List<ButtonUpdate> updates)
    {
        this.activityId = activityId;
        this.updates = updates;
    }

    @Override
    public ToolResult execute(ToolContext toolContext )
    {
        if (activityId == null || activityId.isBlank() || 
            updates == null || updates.isEmpty())
        {
            return new ToolResult(
                false,
                "Error: Invalid payload. Expecting \"activityId\" and a non-empty \"updates\" array."
            );
        }

        Process process = toolContext.getProcess();
        Activity activity = process.getActivities().stream()
            .filter(act -> act.getId().equals(activityId))
            .findFirst()
            .orElse(null);

        if (activity == null)
        {
            return new ToolResult(
                false,
                "Error: Activity with activityId: '" + activityId + "' doesn't exist!."
            );
        }

        List<ButtonUpdateOutput> updatedButtons = new ArrayList<>();
        List<String> errors = new ArrayList<>();

        for (ButtonUpdate update : updates)
        {
            String acceptButtonId = update.getAcceptButtonId();
            String newAcceptButtonName = update.getNewAcceptButtonName();

            if (acceptButtonId == null || acceptButtonId.isBlank() || 
                newAcceptButtonName == null || newAcceptButtonName.isBlank())
            {
                errors.add("Error: Invalid update object. Expecting \"acceptButtonId\" and \"newAcceptButtonName\".");
                continue;
            }

            Optional<AcceptButton> acceptButtonOptional = activity.getAcceptButtons().stream()
                .filter(button -> button.getId().equals(acceptButtonId))
                .findFirst();

            if (acceptButtonOptional.isEmpty())
            {
                errors.add("Error: Accept button with acceptButtonId: '" + acceptButtonId + 
                          "' not found on activity with activityId: '" + activityId + "'.");
                continue;
            }

            AcceptButton acceptButton = acceptButtonOptional.get();
            String oldAcceptButtonId = acceptButton.getId();

            // Generate new ID based on new name
            String newAcceptButtonId = generateUniqueAcceptButtonId(
                activity, 
                newAcceptButtonName, 
                oldAcceptButtonId
            );

            acceptButton.setId(newAcceptButtonId);
            acceptButton.setName(newAcceptButtonName);

            updatedButtons.add(new ButtonUpdateOutput(
                activityId, 
                oldAcceptButtonId, 
                newAcceptButtonId, 
                newAcceptButtonName
            ));

            // TODO: Update button IDs in gateway transition conditions
        }

        String data = "";
        if (updatedButtons.size() > 0)
        {
            String formattedUpdated = updatedButtons.stream()
                .map(u -> "Old acceptButtonId: '" + u.oldAcceptButtonId() + 
                         "', New acceptButtonName: '" + u.newAcceptButtonName() + 
                         "', New acceptButtonId: '" + u.newAcceptButtonId() + "'.")
                .reduce("", (acc, item) -> acc.isEmpty() ? item : acc + "\n" + item);
            data += "Successfully updated " + updatedButtons.size() + 
                   " buttons on activity '" + activityId + "':\n" + formattedUpdated;
        }

        if (errors.size() > 0)
        {
            if (!data.isEmpty())
            {
                data += "\n\n";
            }
            data += "Errors:\n" + String.join("\n", errors);
        }

        if (updatedButtons.isEmpty() && errors.size() > 0)
        {
            return new ToolResult(false, data);
        }

        return new ToolResult(true, data.isEmpty() ? "No buttons were updated." : data);
    }

    private String generateUniqueAcceptButtonId(
        Activity activity, 
        String buttonName, 
        String currentButtonId)
    {
        String baseAcceptButtonId = Normalizer.normalize(buttonName, Normalizer.Form.NFD)
            .toLowerCase()
            .replaceAll("[\\u0300-\\u036f]", "")
            .replaceAll("\\s+", "_")
            .replaceAll("[^a-z0-9_]", "");
        
        if (baseAcceptButtonId.length() > 16)
        {
            baseAcceptButtonId = baseAcceptButtonId.substring(0, 16);
        }

        final String baseId = baseAcceptButtonId;
        String newAcceptButtonId = baseId;
        int counter = 1;

        while (true)
        {
            final String currentId = newAcceptButtonId;
            boolean exists = activity.getAcceptButtons().stream()
                .anyMatch(b -> b.getId().equals(currentId) && !b.getId().equals(currentButtonId));
            if (!exists)
            {
                break;
            }
            String counterStr = String.valueOf(counter);
            int maxLength = 16 - counterStr.length();
            String trimmedId = baseId.length() > maxLength ? 
                              baseId.substring(0, maxLength) : baseId;
            newAcceptButtonId = trimmedId + counter;
            counter++;
        }

        return newAcceptButtonId;
    }

    @Getter
    @Setter
    @AllArgsConstructor
    @NoArgsConstructor
    public static class ButtonUpdate
    {
        @JsonProperty(required = true)
        private String acceptButtonId;

        @JsonProperty(required = true)
        private String newAcceptButtonName;
    }

    public record ButtonUpdateOutput(
        String activityId, 
        String oldAcceptButtonId, 
        String newAcceptButtonId, 
        String newAcceptButtonName)
    {
    }
}
