package com.suncode.pwfl.assistant.agent.transition.tools;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.suncode.pwfl.assistant.AgentType;
import com.suncode.pwfl.assistant.tools.Tool;
import com.suncode.pwfl.assistant.tools.ToolContext;
import com.suncode.pwfl.assistant.tools.ToolDefinition;
import com.suncode.pwfl.assistant.tools.ToolResult;
import com.suncode.pwfl.workflow.process.map.Activity;
import com.suncode.pwfl.workflow.process.map.Process;
import com.suncode.pwfl.workflow.process.map.query.ProcessMapDataQuery;
import com.suncode.pwfl.workflow.process.map.transition.Gateway;
import com.suncode.pwfl.workflow.process.map.transition.GatewayType;
import com.suncode.pwfl.workflow.process.map.transition.Transition;
import com.suncode.pwfl.workflow.process.map.transition.condition.Condition;
import com.suncode.pwfl.workflow.process.map.validator.ConditionValidator;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.UUID;

/**
 * Tool for setting a gateway as the outgoing transition from a source activity.
 * Supports both XOR (exclusive) and AND (parallel) gateways with conditional logic.
 */
@Getter
public class SetGatewayAsOutgoingTransitionFromSourceActivityIdTool implements Tool
{
    public static final ToolDefinition DEFINITION = new ToolDefinition(
        AgentType.TRANSITION,
        "set_gateway_as_outgoing_transition_from_source_activityId",
        """
            Sets a gateway as the outgoing transition from a source activity. If an outgoing transition already exists for the source activity (either direct or to another gateway), it will be overwritten. The gateway defines logic for forking the process flow.
            - 'gatewayType': Can be 'XOR' for an exclusive choice (first condition that is true is taken and process flows to the defined targetActivityId - order of conditions is important then) or 'AND' for a parallel split (multiple paths can be taken simultaneously).
            - 'transitions': An array of objects, where each object defines a path from the gateway to a 'targetActivityId'.
            Each transition is governed by a 'condition' object.
            - For 'XOR' gateways, every transition MUST have a condition! (condition created using acceptButtonId or variableId or both).
            - For 'AND' gateways, the 'condition' for each transition is optional. A transition can be unconditional, or it can have a simple or complex nested condition just like in an 'XOR' gateway. All paths whose conditions are met (including all unconditional paths) will be activated in parallel creating instances of activities for which the conditions have been met.
            The 'condition' object can be a simple 'SimpleCondition' or a complex 'GroupCondition'.
            Each 'condition' object MUST contain 'type' property determining condition type.
            - 'SimpleCondition': A single logical check. It can contain:
                - 'acceptButtonId': The acceptButtonId of an existing button on the source activity. The condition is met if this button was used to complete the task.
                - 'variableId', 'operator', 'value': A comparison of a process variable (e.g., amount > 1000).
                - Both 'acceptButtonId' and a variable check can be combined in one atomic condition.
                - 'type' property MUST be set to 'simple'
            - 'GroupCondition': For complex logic, it contains:
                - 'operator': 'AND' or 'OR'.
                - 'conditions': An array of 'SimpleCondition' and/or other 'GroupCondition' objects, allowing for deep nesting.
                - 'type' property MUST be set to 'group'
            """,
        "To create a gateway, write { \"sourceActivityId\": \"<id>\", \"gatewayType\": \"XOR|AND\", \"transitions\": [ { \"targetActivityId\": \"<id>\", \"condition\": { ... } } ] }. The 'condition' can be a simple check or a nested structure. Example of a complex condition: { \"type\": \"group\", \"operator\": \"OR\", \"conditions\": [ { \"type\": \"group\", \"operator\": \"AND\", \"conditions\": [ { \"type\": \"simple\", \"acceptButtonId\": \"accept\" }, { \"type\": \"simple\", \"variableId\": \"amount\", \"operator\": \">\", \"value\": 1000 } ] }, { \"type\": \"simple\", \"variableId\": \"status\", \"operator\": \"==\", \"value\": \"approved\" } ] }. For 'AND' transitions, provide `{ \"type\": \"group\", \"operator\": \"AND\", \"conditions\": [] }` for unconditional paths. Supported operators: '==', '!=', '<=', '>=', '<', '>'",
        SetGatewayAsOutgoingTransitionFromSourceActivityIdTool.class
    );

    private final String sourceActivityId;
    private final GatewayType gatewayType;
    private final List<TransitionToAdd> transitions;

    @JsonCreator
    public SetGatewayAsOutgoingTransitionFromSourceActivityIdTool(
        @JsonProperty("sourceActivityId") String sourceActivityId,
        @JsonProperty("gatewayType") GatewayType gatewayType,
        @JsonProperty("transitions") List<TransitionToAdd> transitions)
    {
        this.sourceActivityId = sourceActivityId;
        this.gatewayType = gatewayType;
        this.transitions = transitions;
    }

    @Override
    public ToolResult execute(ToolContext toolContext )
    {
        if (sourceActivityId == null || sourceActivityId.isBlank() || 
            gatewayType == null || 
            transitions == null || transitions.isEmpty())
        {
            return new ToolResult(
                false,
                "Error: Invalid payload. Expecting \"sourceActivityId\", \"gatewayType\", and a non-empty \"transitions\" array."
            );
        }

        Process process = toolContext.getProcess();
        ProcessMapDataQuery processMapDataQuery = new ProcessMapDataQuery(process);

        Optional<Activity> sourceActivityOpt = processMapDataQuery.getActivity(sourceActivityId);
        if (sourceActivityOpt.isEmpty())
        {
            return new ToolResult(
                false,
                "Error: Source activity with activityId: '" + sourceActivityId + "' doesn't exist!."
            );
        }

        Activity sourceActivity = sourceActivityOpt.get();
        List<String> errors = new ArrayList<>();
        List<Transition> validatedTransitions = new ArrayList<>();

        for (TransitionToAdd transitionToAdd : transitions)
        {
            String targetActivityId = transitionToAdd.getTargetActivityId();
            Optional<Activity> targetActivityOpt = processMapDataQuery.getActivity(targetActivityId);

            if (targetActivityOpt.isEmpty())
            {
                errors.add("Target activity with activityId: '" + targetActivityId + "' not found.");
                continue;
            }

            Condition condition = transitionToAdd.getCondition();

            // For XOR gateways, condition is required
            if (gatewayType == GatewayType.XOR && (condition == null))
            {
                errors.add("For XOR gateway, a condition is required for transition to '" + targetActivityId +
                          "'. The transition should be linked to at least a dedicated button if there is no need to define conditions on process variables.");
                continue;
            }

            if (condition != null)
            {
                try
                {
                    ConditionValidator.validateCondition(condition, process, sourceActivity);
                }
                catch (Exception e)
                {
                    errors.add(e.getMessage());
                    continue;
                }
            }

            Transition transition = new Transition(
                UUID.randomUUID().toString(), 
                "",
                targetActivityId, 
                condition
            );
            validatedTransitions.add(transition);
        }

        if (!errors.isEmpty())
        {
            return new ToolResult(false, "Errors found:\n" + String.join("\n", errors));
        }

        // Remove existing outgoing transitions from source activity (overwrite behavior)
        process.getTransitions().removeIf(transition -> 
            transition.getSourceId().equals(sourceActivityId));

        // Create gateway
        Gateway gateway = new Gateway(UUID.randomUUID().toString(), gatewayType);
        process.getGateways().add(gateway);

        // Create transition from source activity to gateway
        Transition gatewayTransition = new Transition(
            UUID.randomUUID().toString(), 
            sourceActivityId, 
            gateway.getId()
        );
        process.getTransitions().add(gatewayTransition);

        // Create transitions from gateway to target activities
        for (Transition transition : validatedTransitions)
        {
            transition.setSourceId(gateway.getId());
            process.getTransitions().add(transition);
        }

        return new ToolResult(
            true,
            "Successfully set " + gatewayType + 
            " gateway as the outcoming transition from '" + sourceActivity.getName() + "'."
        );
    }

    @Getter
    @Setter
    @AllArgsConstructor
    @NoArgsConstructor
    public static class TransitionToAdd
    {
        @JsonProperty(required = true)
        private String targetActivityId;

        @JsonProperty
        private Condition condition;
    }
}
