package com.suncode.pwfl.assistant.agent.transition.tools;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.suncode.pwfl.assistant.AgentType;
import com.suncode.pwfl.assistant.tools.Tool;
import com.suncode.pwfl.assistant.tools.ToolContext;
import com.suncode.pwfl.assistant.tools.ToolDefinition;
import com.suncode.pwfl.assistant.tools.ToolResult;
import com.suncode.pwfl.workflow.process.map.Activity;
import com.suncode.pwfl.workflow.process.map.Process;
import com.suncode.pwfl.workflow.process.map.transition.Transition;
import lombok.Getter;

import java.util.UUID;

/**
 * Tool for setting a direct, unconditional transition from source to target activity.
 * Overwrites any existing outgoing transition from the source activity.
 */
@Getter
public class SetActivityAsOutgoingTransitionFromSourceActivityIdTool implements Tool
{
    public static final ToolDefinition DEFINITION = new ToolDefinition(
        AgentType.TRANSITION,
        "set_activity_as_outgoing_transition_from_source_activityId",
        """
            Sets a direct, unconditional transition from a source activity to a target activity.
            If an outgoing transition already exists for the source activity (either direct or to a gateway), it will be overwritten.
            This is used for simple, direct flows between two activities.
            Returns a confirmation of the created transition: { "sourceActivityId": "<id>", "targetActivityId": "<id>" }
            """,
        "To create a direct transition between activities, write { \"sourceActivityId\": \"<id>\", \"targetActivityId\": \"<id>\" }",
        SetActivityAsOutgoingTransitionFromSourceActivityIdTool.class
    );

    private final String sourceActivityId;
    private final String targetActivityId;

    @JsonCreator
    public SetActivityAsOutgoingTransitionFromSourceActivityIdTool(
        @JsonProperty("sourceActivityId") String sourceActivityId,
        @JsonProperty("targetActivityId") String targetActivityId)
    {
        this.sourceActivityId = sourceActivityId;
        this.targetActivityId = targetActivityId;
    }

    @Override
    public ToolResult execute(ToolContext toolContext )
    {
        if (sourceActivityId == null || sourceActivityId.isBlank() || 
            targetActivityId == null || targetActivityId.isBlank())
        {
            return new ToolResult(
                false,
                "Error: Invalid payload. Expecting \"sourceActivityId\" and \"targetActivityId\"."
            );
        }

        Process process = toolContext.getProcess();

        Activity sourceActivity = process.getActivities().stream()
            .filter(act -> act.getId().equals(sourceActivityId))
            .findFirst()
            .orElse(null);

        if (sourceActivity == null)
        {
            return new ToolResult(
                false,
                "Error: Source activity with activityId: '" + sourceActivityId + "' doesn't exist!."
            );
        }

        Activity targetActivity = process.getActivities().stream()
            .filter(act -> act.getId().equals(targetActivityId))
            .findFirst()
            .orElse(null);

        if (targetActivity == null)
        {
            return new ToolResult(
                false,
                "Error: Target activity with activityId: '" + targetActivityId + "' doesn't exist!."
            );
        }

        // Remove any existing outgoing transitions from source activity
        process.getTransitions().removeIf(transition -> 
            transition.getSourceId().equals(sourceActivityId));

        // Create new direct transition
        Transition transition = new Transition(
            UUID.randomUUID().toString(),
            sourceActivityId,
            targetActivityId
        );
        process.getTransitions().add(transition);

        return new ToolResult(
            true,
            "Successfully set activity '" + targetActivity.getName() + 
            "' as the outcoming transition from '" + sourceActivity.getName() + "'."
        );
    }
}
