package com.suncode.pwfl.assistant.agent.transition.tools;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.suncode.pwfl.assistant.AgentType;
import com.suncode.pwfl.assistant.tools.Tool;
import com.suncode.pwfl.assistant.tools.ToolContext;
import com.suncode.pwfl.assistant.tools.ToolDefinition;
import com.suncode.pwfl.assistant.tools.ToolResult;
import com.suncode.pwfl.workflow.process.map.Activity;
import com.suncode.pwfl.workflow.process.map.Process;
import com.suncode.pwfl.workflow.process.map.transition.Transition;
import lombok.Getter;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * Tool for retrieving outgoing transitions from activities.
 * Shows transition details including types and conditions.
 */
@Getter
public class GetOutgoingTransitionOfActivitiesTool implements Tool
{
    public static final ToolDefinition DEFINITION = new ToolDefinition(
        AgentType.TRANSITION,
        "get_outgoing_transition_of_activities",
        "Gets the details of the outgoing transitions from one or more activities. Tool returns a list of transition details (with their types and conditions of transition to each activity) in the following format: { \"results\": [ { \"activityId\": \"<id>\", \"outcomingTransition\": { ... } } ] }",
        "To get the outcoming transitions of activities, write { \"activityIds\": [\"<id1>\", \"<id2>\"] }",
        GetOutgoingTransitionOfActivitiesTool.class
    );

    private final List<String> activityIds;

    @JsonCreator
    public GetOutgoingTransitionOfActivitiesTool(
        @JsonProperty("activityIds") List<String> activityIds)
    {
        this.activityIds = activityIds;
    }

    @Override
    public ToolResult execute(ToolContext toolContext )
    {
        if (activityIds == null || activityIds.isEmpty())
        {
            return new ToolResult(
                false,
                "Error: Invalid payload. Expecting a non-empty \"activityIds\" array."
            );
        }

        Process process = toolContext.getProcess();
        List<String> results = new ArrayList<>();
        List<String> errors = new ArrayList<>();

        for (String activityId : activityIds)
        {
            Optional<Activity> activityOpt = process.getActivities().stream()
                .filter(act -> act.getId().equals(activityId))
                .findFirst();

            if (activityOpt.isEmpty())
            {
                errors.add("Error: Activity with activityId: '" + activityId + "' doesn't exist!.");
                continue;
            }

            Activity activity = activityOpt.get();
            List<Transition> outgoingTransitions = process.getTransitions().stream()
                .filter(transition -> transition.getSourceId().equals(activityId))
                .toList();

            if (outgoingTransitions.isEmpty())
            {
                results.add("Activity '" + activity.getName() + "' (activityId: " + activityId + 
                          "): No outcoming transition defined.");
            }
            else
            {
                String transitionDetails = outgoingTransitions.stream()
                    .map(t -> "Transition to: " + t.getTargetId() + 
                             (t.getCondition() != null ? 
                                 ", Condition: " + t.getCondition().toConditionText() : ""))
                    .reduce("", (acc, item) -> acc.isEmpty() ? item : acc + ", " + item);
                results.add("Activity '" + activity.getName() + "' (activityId: " + activityId + 
                          "):\n" + transitionDetails);
            }
        }

        String data = "";
        if (results.size() > 0)
        {
            data += "Successfully retrieved outcoming transitions:\n" + String.join("\n\n", results);
        }

        if (errors.size() > 0)
        {
            if (!data.isEmpty())
            {
                data += "\n\n";
            }
            data += "Errors:\n" + String.join("\n", errors);
        }

        if (results.isEmpty() && errors.size() > 0)
        {
            return new ToolResult(false, data);
        }

        return new ToolResult(true, data.isEmpty() ? "No activities processed." : data);
    }
}
