package com.suncode.pwfl.assistant.agent.transition.tools;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.suncode.pwfl.assistant.AgentType;
import com.suncode.pwfl.assistant.tools.Tool;
import com.suncode.pwfl.assistant.tools.ToolContext;
import com.suncode.pwfl.assistant.tools.ToolDefinition;
import com.suncode.pwfl.assistant.tools.ToolResult;
import com.suncode.pwfl.workflow.process.map.Activity;
import com.suncode.pwfl.workflow.process.map.Process;
import lombok.Getter;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * Tool for retrieving accept buttons from activities.
 */
@Getter
public class GetAcceptButtonsOfActivitiesTool implements Tool
{
    public static final ToolDefinition DEFINITION = new ToolDefinition(
        AgentType.TRANSITION,
        "get_accept_buttons_of_activities",
        "Retrieves all accept buttons associated with one or more activities. Returns a list of activities with their buttons in the following format: { \"results\": [ { \"activityId\": \"<id>\", \"acceptButtons\": [ { \"acceptButtonId\": \"<id>\", \"acceptButtonName\": \"<name>\" } ] } ] }",
        "To get all accept buttons for activities, write { \"activityIds\": [\"<id1>\", \"<id2>\"] }",
        GetAcceptButtonsOfActivitiesTool.class
    );

    private final List<String> activityIds;

    @JsonCreator
    public GetAcceptButtonsOfActivitiesTool(
        @JsonProperty("activityIds") List<String> activityIds)
    {
        this.activityIds = activityIds;
    }

    @Override
    public ToolResult execute(ToolContext toolContext )
    {
        if (activityIds == null || activityIds.isEmpty())
        {
            return new ToolResult(
                false,
                "Error: Invalid payload. Expecting a non-empty \"activityIds\" array."
            );
        }

        Process process = toolContext.getProcess();
        List<String> results = new ArrayList<>();
        List<String> errors = new ArrayList<>();

        for (String activityId : activityIds)
        {
            Optional<Activity> activity = process.getActivities().stream()
                .filter(act -> act.getId().equals(activityId))
                .findFirst();

            if (activity.isEmpty())
            {
                errors.add("Error: Activity with activityId: '" + activityId + "' doesn't exist!.");
                continue;
            }

            Activity act = activity.get();
            if (act.getAcceptButtons().isEmpty())
            {
                results.add("Activity '" + act.getName() + "' (ID: " + activityId + 
                          "): No accept buttons found.");
                continue;
            }

            String formattedButtons = "";
            int index = 1;
            for (var button : act.getAcceptButtons())
            {
                if (!formattedButtons.isEmpty())
                {
                    formattedButtons += "\n";
                }
                formattedButtons += "  " + index + ". acceptButtonId: " + button.getId() + 
                                  ", acceptButtonName: " + button.getName();
                index++;
            }
            results.add("Activity '" + act.getName() + "' (activityId: " + activityId + 
                      "):\n" + formattedButtons);
        }

        String data = "";
        if (results.size() > 0)
        {
            data += "Successfully retrieved accept buttons:\n" + String.join("\n\n", results);
        }

        if (errors.size() > 0)
        {
            if (!data.isEmpty())
            {
                data += "\n\n";
            }
            data += "Errors:\n" + String.join("\n", errors);
        }

        if (results.isEmpty() && errors.size() > 0)
        {
            return new ToolResult(false, data);
        }

        return new ToolResult(true, data.isEmpty() ? "No activities processed." : data);
    }
}
