package com.suncode.pwfl.assistant.agent.transition.tools;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.suncode.pwfl.assistant.AgentType;
import com.suncode.pwfl.assistant.tools.Tool;
import com.suncode.pwfl.assistant.tools.ToolContext;
import com.suncode.pwfl.assistant.tools.ToolDefinition;
import com.suncode.pwfl.assistant.tools.ToolResult;
import com.suncode.pwfl.workflow.process.map.Activity;
import com.suncode.pwfl.workflow.process.map.Process;
import com.suncode.pwfl.workflow.process.map.query.ProcessMapDataQuery;
import com.suncode.pwfl.workflow.process.map.transition.Gateway;
import com.suncode.pwfl.workflow.process.map.transition.Transition;
import lombok.Getter;

import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

/**
 * Tool for deleting specific transition paths from a gateway.
 * Identified by target activity IDs.
 */
@Getter
public class DeleteTransitionPathsFromGatewayTool implements Tool
{
    public static final ToolDefinition DEFINITION = new ToolDefinition(
        AgentType.TRANSITION,
        "delete_transition_paths_from_gateway",
        "Deletes one or more existing outgoing transitions from a source activity's gateway, identified by their target activityIds. Fails if the outgoing transition is not a gateway. Returns a confirmation of the deletion: { \"sourceActivityId\": \"<id>\", \"deletedTargetActivityIds\": [\"<id1>\", \"<id2>\"] }",
        "To delete transitions from a gateway, write { \"sourceActivityId\": \"<id>\", \"targetActivityIds\": [\"<id1>\", \"<id2>\"] }",
        DeleteTransitionPathsFromGatewayTool.class
    );

    private final String sourceActivityId;
    private final List<String> targetActivityIds;

    @JsonCreator
    public DeleteTransitionPathsFromGatewayTool(
        @JsonProperty("sourceActivityId") String sourceActivityId,
        @JsonProperty("targetActivityIds") List<String> targetActivityIds)
    {
        this.sourceActivityId = sourceActivityId;
        this.targetActivityIds = targetActivityIds;
    }

    @Override
    public ToolResult execute(ToolContext toolContext )
    {
        if (sourceActivityId == null || sourceActivityId.isBlank() || 
            targetActivityIds == null || targetActivityIds.isEmpty())
        {
            return new ToolResult(
                false,
                "Error: Invalid payload. Expecting \"sourceActivityId\" and a non-empty \"targetActivityIds\" array."
            );
        }

        Process process = toolContext.getProcess();
        ProcessMapDataQuery processMapDataQuery = new ProcessMapDataQuery(process);

        Optional<Activity> activityOpt = processMapDataQuery.getActivity(sourceActivityId);
        if (activityOpt.isEmpty())
        {
            return new ToolResult(
                false,
                "Error: Source activity with activityId: '" + sourceActivityId + "' doesn't exist!."
            );
        }

        Optional<Gateway> gatewayOpt = processMapDataQuery.getGatewayTransitionFromActivity(sourceActivityId);
        if (gatewayOpt.isEmpty())
        {
            return new ToolResult(
                false,
                "Error: The outgoing transition from activity '" + sourceActivityId + "' is not a gateway."
            );
        }

        Gateway gateway = gatewayOpt.get();
        Set<String> uniqueTargetIdsToDelete = new HashSet<>(targetActivityIds);
        List<Transition> gatewayOutcomingTransitions = 
            processMapDataQuery.getOutgoingTransitionsFromGateway(gateway.getId());

        List<String> transitionsToDelete = gatewayOutcomingTransitions.stream()
            .filter(transition -> uniqueTargetIdsToDelete.contains(transition.getTargetId()))
            .map(Transition::getId)
            .toList();

        process.getTransitions().removeIf(transition -> transitionsToDelete.contains(transition.getId()));
        
        int deletedCount = transitionsToDelete.size();

        if (deletedCount == 0)
        {
            return new ToolResult(
                false,
                "Error: No transitions to target activities [" + String.join(", ", targetActivityIds) + 
                "] were found in the gateway of source activity '" + sourceActivityId + "'."
            );
        }

        return new ToolResult(
            true,
            "Successfully deleted " + deletedCount + 
            " transition(s) from the gateway of source activity '" + sourceActivityId + 
            "'. Deleted targets: [" + String.join(", ", transitionsToDelete) + "]."
        );
    }
}
