package com.suncode.pwfl.assistant.agent.transition.tools;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.suncode.pwfl.assistant.AgentType;
import com.suncode.pwfl.assistant.tools.Tool;
import com.suncode.pwfl.assistant.tools.ToolContext;
import com.suncode.pwfl.assistant.tools.ToolDefinition;
import com.suncode.pwfl.assistant.tools.ToolResult;
import com.suncode.pwfl.workflow.process.map.Activity;
import com.suncode.pwfl.workflow.process.map.Process;
import lombok.Getter;

import java.util.Optional;

/**
 * Tool for deleting the entire outgoing transition from an activity.
 * Works for both direct transitions and gateways.
 */
@Getter
public class DeleteOutgoingTransitionFromActivityTool implements Tool
{
    public static final ToolDefinition DEFINITION = new ToolDefinition(
        AgentType.TRANSITION,
        "delete_outgoing_transition_from_activity",
        "Deletes the entire outgoing transition from a specified activity, whether it is a direct transition or a gateway. Returns a confirmation of the deletion: { \"sourceActivityId\": \"<id>\" }",
        "To delete the entire outgoing transition from an activity, write { \"sourceActivityId\": \"<id>\" }",
        DeleteOutgoingTransitionFromActivityTool.class
    );

    private final String sourceActivityId;

    @JsonCreator
    public DeleteOutgoingTransitionFromActivityTool(
        @JsonProperty("sourceActivityId") String sourceActivityId)
    {
        this.sourceActivityId = sourceActivityId;
    }

    @Override
    public ToolResult execute(ToolContext toolContext )
    {
        if (sourceActivityId == null || sourceActivityId.isBlank())
        {
            return new ToolResult(false, "Error: Invalid payload. Expecting \"sourceActivityId\".");
        }

        Process process = toolContext.getProcess();

        Optional<Activity> activityOpt = process.getActivities().stream()
            .filter(act -> act.getId().equals(sourceActivityId))
            .findFirst();

        if (activityOpt.isEmpty())
        {
            return new ToolResult(
                false,
                "Error: Source activity with activityId: '" + sourceActivityId + "' doesn't exist!."
            );
        }

        long initialCount = process.getTransitions().stream()
            .filter(transition -> transition.getSourceId().equals(sourceActivityId))
            .count();

        if (initialCount == 0)
        {
            return new ToolResult(
                false,
                "Error: Activity with activityId: '" + sourceActivityId + 
                "' does not have an outgoing transition to delete."
            );
        }

        process.getTransitions().removeIf(transition -> 
            transition.getSourceId().equals(sourceActivityId));

        return new ToolResult(
            true,
            "Successfully deleted outgoing transition from activity with activityId: '" + 
            sourceActivityId + "'."
        );
    }
}
