package com.suncode.pwfl.assistant.agent.transition.tools;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.suncode.pwfl.assistant.AgentType;
import com.suncode.pwfl.assistant.tools.Tool;
import com.suncode.pwfl.assistant.tools.ToolContext;
import com.suncode.pwfl.assistant.tools.ToolDefinition;
import com.suncode.pwfl.assistant.tools.ToolResult;
import com.suncode.pwfl.workflow.process.map.AcceptButton;
import com.suncode.pwfl.workflow.process.map.Activity;
import com.suncode.pwfl.workflow.process.map.Process;
import lombok.Getter;

import java.util.ArrayList;
import java.util.List;

/**
 * Tool for deleting accept buttons from an activity.
 */
@Getter
public class DeleteAcceptButtonsOnActivityTool implements Tool
{
    public static final ToolDefinition DEFINITION = new ToolDefinition(
        AgentType.TRANSITION,
        "delete_accept_buttons_on_activity",
        "Deletes one or more accept buttons from a specified activity. Returns the IDs of the deleted buttons in the following format: { \"activityId\": \"<id>\", \"deletedAcceptButtonIds\": [\"<id1>\", \"<id2>\"] }",
        "To delete accept buttons, write { \"activityId\": \"<id>\", \"acceptButtonIds\": [\"<id1>\", \"<id2>\"] }",
        DeleteAcceptButtonsOnActivityTool.class
    );

    private final String activityId;
    private final List<String> acceptButtonIds;

    @JsonCreator
    public DeleteAcceptButtonsOnActivityTool(
        @JsonProperty("activityId") String activityId,
        @JsonProperty("acceptButtonIds") List<String> acceptButtonIds)
    {
        this.activityId = activityId;
        this.acceptButtonIds = acceptButtonIds;
    }

    @Override
    public ToolResult execute(ToolContext toolContext )
    {
        if (activityId == null || activityId.isBlank() || 
            acceptButtonIds == null || acceptButtonIds.isEmpty())
        {
            return new ToolResult(
                false,
                "Error: Invalid payload. Expecting \"activityId\" and a non-empty \"acceptButtonIds\" array."
            );
        }

        Process process = toolContext.getProcess();
        Activity activity = process.getActivities().stream()
            .filter(act -> act.getId().equals(activityId))
            .findFirst()
            .orElse(null);

        if (activity == null)
        {
            return new ToolResult(
                false,
                "Error: Activity with activityId: '" + activityId + "' doesn't exist!."
            );
        }

        List<String> deletedButtonIds = new ArrayList<>();
        List<String> notFoundButtonIds = new ArrayList<>();

        for (String acceptButtonId : acceptButtonIds)
        {
            int buttonIndex = -1;
            for (int i = 0; i < activity.getAcceptButtons().size(); i++)
            {
                AcceptButton button = activity.getAcceptButtons().get(i);
                if (button.getId().equals(acceptButtonId))
                {
                    buttonIndex = i;
                    break;
                }
            }

            if (buttonIndex == -1)
            {
                notFoundButtonIds.add(acceptButtonId);
                continue;
            }

            activity.getAcceptButtons().remove(buttonIndex);
            deletedButtonIds.add(acceptButtonId);

            // TODO: Clean up button references from gateway transition conditions
        }

        String data = "";
        if (deletedButtonIds.size() > 0)
        {
            data += "Successfully deleted " + deletedButtonIds.size() + 
                   " buttons from activity '" + activityId + "': " + 
                   String.join(", ", deletedButtonIds) + ".";
        }

        if (notFoundButtonIds.size() > 0)
        {
            if (!data.isEmpty())
            {
                data += "\n";
            }
            data += "Buttons not found on activity '" + activityId + "': " + 
                   String.join(", ", notFoundButtonIds) + ".";
        }

        if (deletedButtonIds.isEmpty() && !notFoundButtonIds.isEmpty())
        {
            return new ToolResult(false, data);
        }

        return new ToolResult(true, data.isEmpty() ? "No buttons were deleted." : data);
    }
}
