package com.suncode.pwfl.assistant.agent.transition.tools;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.suncode.pwfl.assistant.AgentType;
import com.suncode.pwfl.assistant.tools.Tool;
import com.suncode.pwfl.assistant.tools.ToolContext;
import com.suncode.pwfl.assistant.tools.ToolDefinition;
import com.suncode.pwfl.assistant.tools.ToolResult;
import com.suncode.pwfl.workflow.process.map.Activity;
import com.suncode.pwfl.workflow.process.map.Process;
import com.suncode.pwfl.workflow.process.map.query.ProcessMapDataQuery;
import com.suncode.pwfl.workflow.process.map.transition.Gateway;
import com.suncode.pwfl.workflow.process.map.transition.Transition;
import com.suncode.pwfl.workflow.process.map.transition.condition.Condition;
import com.suncode.pwfl.workflow.process.map.validator.ConditionValidator;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

import java.util.Optional;
import java.util.UUID;

/**
 * Tool for appending a new conditional transition to an existing gateway.
 * Fails if the source activity's outgoing transition is not a gateway.
 */
@Slf4j
@Getter
public class AppendOutgoingTransitionToActivityTool implements Tool
{
    public static final ToolDefinition DEFINITION = new ToolDefinition(
        AgentType.TRANSITION,
        "append_outgoing_transition_to_activity",
        "Appends a new conditional transition to an existing gateway of a source activity. This tool fails if the source activity's outgoing transition is not a gateway. The condition can be a simple check or a complex nested structure involving `acceptButtonId` and variables. Returns the details of the appended transition: { \"sourceActivityId\": \"<id>\", \"appendedTransition\": { \"targetActivityId\": \"<id>\", \"condition\": { ... } } }",
        "To append a transition to a gateway, write { \"sourceActivityId\": \"<id>\", \"targetActivityId\": \"<id>\", \"condition\": { ... } }. The 'condition' is required and can be a complex structure. See 'set_gateway_as_outgoing_transition_from_source_activityId' for condition examples.",
        AppendOutgoingTransitionToActivityTool.class
    );

    private final String sourceActivityId;
    private final String targetActivityId;
    private final Condition condition;

    @JsonCreator
    public AppendOutgoingTransitionToActivityTool(
        @JsonProperty("sourceActivityId") String sourceActivityId,
        @JsonProperty("targetActivityId") String targetActivityId,
        @JsonProperty("condition") Condition condition)
    {
        this.sourceActivityId = sourceActivityId;
        this.targetActivityId = targetActivityId;
        this.condition = condition;
    }

    @Override
    public ToolResult execute(ToolContext toolContext )
    {
        if (sourceActivityId == null || sourceActivityId.isBlank() || 
            targetActivityId == null || targetActivityId.isBlank() || 
            condition == null)
        {
            return new ToolResult(
                false,
                "Error: Invalid payload. Expecting \"sourceActivityId\", \"targetActivityId\" and \"condition\"."
            );
        }

        Process process = toolContext.getProcess();
        ProcessMapDataQuery processMapDataQuery = new ProcessMapDataQuery(process);

        Optional<Activity> sourceActivityOpt = processMapDataQuery.getActivity(sourceActivityId);
        if (sourceActivityOpt.isEmpty())
        {
            return new ToolResult(
                false,
                "Error: Source activity with activityId: '" + sourceActivityId + "' doesn't exist!."
            );
        }

        Optional<Gateway> gatewayOpt = processMapDataQuery.getGatewayTransitionFromActivity(sourceActivityId);
        if (gatewayOpt.isEmpty())
        {
            return new ToolResult(
                false,
                "Error: Source activity's with activityId: '" + sourceActivityId + 
                "' outcoming transition is not a gateway."
            );
        }

        Optional<Activity> targetActivityOpt = processMapDataQuery.getActivity(targetActivityId);
        if (targetActivityOpt.isEmpty())
        {
            return new ToolResult(
                false,
                "Error: Target activity with activityId: '" + targetActivityId + "' doesn't exist!."
            );
        }

        // Validate condition
        if (condition != null)
        {
            try
            {
                ConditionValidator.validateCondition(condition, process, sourceActivityOpt.get());
            }
            catch (Exception e)
            {
                return new ToolResult(false, e.getMessage());
            }
        }

        Gateway gateway = gatewayOpt.get();
        Transition newTransition = new Transition(
            UUID.randomUUID().toString(),
            gateway.getId(),
            targetActivityId,
            condition
        );
        process.getTransitions().add(newTransition);

        return new ToolResult(
            true,
            "Successfully appended transition to gateway. sourceActivityId: '" + sourceActivityId + 
            "', targetActivityId: '" + targetActivityId + "'."
        );
    }
}
