package com.suncode.pwfl.assistant.agent.transition.prompts;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.suncode.config.AIAssistantObjectMapper;
import com.suncode.pwfl.assistant.model.Action;
import com.suncode.pwfl.assistant.model.AgentState;
import com.suncode.pwfl.assistant.model.Task;
import com.suncode.pwfl.assistant.prompting.PromptProvider;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;

import java.util.List;
import java.util.stream.Collectors;

@RequiredArgsConstructor
public class TransitionAgentAnswerPromptProvider
    implements PromptProvider
{

    @SneakyThrows
    public String provide( AgentState state )
    {
        ObjectMapper objectMapper = new AIAssistantObjectMapper();

        String prompt = """
            - Inform the agent about the tasks you've performed. Be concise but informative.
            - Use the same language as the agent's message to answer.
            - Don't ask about further actions.
            - Never tell the agent what tool names you used, if you used a tool that does not exist, just ignore this fact and inform the agent that you can't answer to this question/can't perform this task.
            - Omit information about what was verified to achieve the goal (check/get methods) from the response. In the response, we're interested in the result, not the individual steps—the agent doesn't need to know about it.
            - If the agent asked for information about some components - provide full information about them (including their ids like buttonId, transitionId).
            
            <performed_tasks>
            %s
            </performed_tasks>
            """;

        String performedTasks = state.getTasks().stream()
            .map( task -> buildTaskTag( task, objectMapper ) )
            .collect( Collectors.joining( "\n" ) );

        return String.format( prompt, performedTasks );
    }

    private String buildTaskTag( Task task, ObjectMapper objectMapper )
    {
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append( String.format(
            "<task name=\"%s\" status=\"%s\">",
            task.getName(), task.getState().name()
        ) );
        stringBuilder.append( String.format(
            "  <description>%s</description>", task.getDescription()
        ) );
        stringBuilder.append( "  <actions>" );
        stringBuilder.append( buildActionsTagContent( task.getActions(), objectMapper ) );
        stringBuilder.append( "  </actions>" );
        return stringBuilder.toString();
    }

    @SneakyThrows
    private String buildActionsTagContent( List<Action> actions, ObjectMapper objectMapper )
    {
        return actions.stream()
            .map( action -> String.format( "  %s", buildActionTag( action, objectMapper ) ) )
            .collect( Collectors.joining( "\n" ) );
    }

    @SneakyThrows
    private String buildActionTag( Action action, ObjectMapper objectMapper )
    {
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append( String.format(
            "<action tool=\"%s\" status=\"%s\">",
            action.getToolName(), action.getState().name()
        ) );

        if ( action.getResult() != null )
        {
            stringBuilder.append( String.format(
                "  <result>%s</result>",
                action.getResult().getData() instanceof String
                    ? action.getResult().getData()
                    : objectMapper.writeValueAsString( action.getResult().getData() )
            ) );
        }

        stringBuilder.append( "<action>" );
        return stringBuilder.toString();
    }

}
