package com.suncode.pwfl.assistant.agent.main.tools;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.suncode.pwfl.assistant.AgentContext;
import com.suncode.pwfl.assistant.AgentExecutor;
import com.suncode.pwfl.assistant.AgentType;
import com.suncode.pwfl.assistant.model.ActionResult;
import com.suncode.pwfl.assistant.tools.Tool;
import com.suncode.pwfl.assistant.tools.ToolContext;
import com.suncode.pwfl.assistant.tools.ToolDefinition;
import com.suncode.pwfl.assistant.tools.ToolResult;
import com.suncode.pwfl.assistant.agent.transition.TransitionAgent;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

/**
 * Tool that delegates to the Transitions specialized agent.
 * This agent handles process flow, transitions, acceptance buttons, and conditions.
 */
@Slf4j
@Getter
public class TransitionsAgentTool implements Tool
{
    public static final ToolDefinition DEFINITION = new ToolDefinition(
        AgentType.MAIN,
        "transitions_agent",
        "This tool is a specialized LLM agent for managing all aspects of the process flow. It is responsible for creating, modifying, and deleting transitions between activities, any acceptance buttons/buttons, and the logical conditions that govern the process path. You MUST delegate any user request related to the process flow/acceptance buttons/buttons and the logical conditions to this agent. The agent's capabilities include:\n" +
        "            - **Acceptance Buttons/Buttons**: Creating, updating, deleting, checking existence of acceptance buttons/buttons. These buttons can be used as triggers in transition conditions.\n" +
        "            - **Direct Transitions**: Creating a simple, unconditional path from one activity to another.\n" +
        "            - **Gateway Transitions**: Setting up complex conditional flows from a source activity using gateways:\n" +
        "            - **XOR Gateway**: For exclusive paths, where the process follows the first path whose condition is met. Conditions are MANDATORY for all paths.\n" +
        "            - **AND Gateway**: For parallel paths, where the process follows all paths whose conditions are met. Conditions are optional; paths without conditions are always taken.\n" +
        "            - **Complex Conditions**: Building simple or deeply nested logical conditions for gateway transitions. Conditions can be based on which acceptance button was pressed, the values of process variables (e.g., amount > 1000), or a combination of both using AND/OR logic.\n" +
        "            - **Modifying Transitions**: Adding new conditional paths to an existing gateway.\n" +
        "            - **Deleting Transitions**: Removing specific paths from a gateway or deleting the entire outgoing transition structure from an activity.\n" +
        "            - **Ending the Process**: Designating a specific activity as the final step of the process.\n" +
        "        \n" +
        "            You MUST provide single complete instruction that represents a full logical task that is related to the process flow if you want to delegate the task to this agent. Avoid splitting the task into multiple smaller tasks (if possible) - if agent will get a complete instruction, he will be able to handle it properly and he will plan what to do with it by himself. After work is done, agent willinform you about results of your query.",
        "To invoke the agent, write { \"query\": \"<your query>\" }. Your query should be a clear and complete instruction that represents a full logical task that is related to the process flow. Avoid splitting the task into multiple smaller queries (e.g., first checking if an element exists, then acting on it). The agent can handle creating necessary elements (like buttons) and linking them in a single operation. For example: \"From activity 'register_invoice', create 'Approve' and 'Reject' buttons. If 'Approve' is clicked, transition to 'approve_invoice'. If 'Reject' is clicked, end the process.\" Provide all necessary details in the query, such as:\n- Source and target activityIds.\n- Names for any new acceptance buttons or mentioned buttons in the user query.\n- Detailed conditions for transitions, including variableIds, operators, and values if applicable.",
        TransitionsAgentTool.class
    );

    private final String query;

    @JsonCreator
    public TransitionsAgentTool(@JsonProperty("query") String query)
    {
        this.query = query;
    }

    @Override
    public ToolResult execute(ToolContext toolContext )
    {
        if (query == null || query.isBlank())
        {
            return new ToolResult(false, "Error: Invalid payload. Expecting a non-empty \"query\" string.");
        }

        if ( toolContext.getLlmService() == null)
        {
            return new ToolResult(false, "Error: LlmService not available in context.");
        }

        try
        {
            AgentContext context = toolContext.getContext();

            TransitionAgent transitionAgent = new TransitionAgent( context, toolContext.getProcess() );
            AgentExecutor agentExecutor = new AgentExecutor( context );

            agentExecutor.execute( transitionAgent, query );
            
            ActionResult finalAnswer = transitionAgent.getState().getFinalAnswer();
            String response = finalAnswer != null ? 
                            finalAnswer.getData().toString() : 
                            "Could not return final answer";
            
            return new ToolResult(true, response);
        }
        catch (Exception e)
        {
            log.error("Failed to execute transitions agent", e);
            return new ToolResult(false, "Error executing agent: " + e.getMessage());
        }
    }
}
