package com.suncode.pwfl.assistant.agent.main.prompts;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.suncode.config.AIAssistantObjectMapper;
import com.suncode.pwfl.assistant.model.Action;
import com.suncode.pwfl.assistant.model.AgentState;
import com.suncode.pwfl.assistant.model.Task;
import com.suncode.pwfl.assistant.prompting.PromptProvider;
import lombok.SneakyThrows;

import java.util.List;
import java.util.stream.Collectors;

public class MainAgentAnswerPromptProvider
    implements PromptProvider
{
    @SneakyThrows
    public String provide( AgentState state ){
        ObjectMapper objectMapper = new AIAssistantObjectMapper();

        String prompt = """
            - Your primary goal is to provide a brief, friendly confirmation that the user's request has been completed.
            - The response should be a short, reassuring message in a natural, conversational tone.
            - Always use the same language as the user's original message.
            - **Do not** list the individual tasks or actions performed.
            - Omit technical details from the response:
                - Never mention the names of the tools used.
                - Do not describe intermediate verification steps (like check/get methods).
                - Use full component names instead of their internal IDs.
            - Do not ask the user for next steps or further actions.
            - If a task could not be completed, simply state that it wasn't possible, without mentioning details.
            
            <performed_tasks>
            %s
            </performed_tasks>
            """;

        String performedTasks = state.getTasks().stream()
            .map( task -> buildTaskTag( task, objectMapper ) )
            .collect( Collectors.joining( "\n" ) );

        return String.format( prompt, performedTasks );
    }

    private String buildTaskTag( Task task, ObjectMapper objectMapper )
    {
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append( String.format(
            "<task name=\"%s\" status=\"%s\">",
            task.getName(), task.getState().name()
        ) );
        stringBuilder.append( String.format(
            "  <description>%s</description>",
            task.getDescription()
        ) );
        stringBuilder.append( "  <actions>" );
        stringBuilder.append( buildActionsTagContent( task.getActions(), objectMapper ) );
        stringBuilder.append( "  </actions>" );
        return stringBuilder.toString();
    }

    @SneakyThrows
    private String buildActionsTagContent( List<Action> actions, ObjectMapper objectMapper )
    {
        return actions.stream()
            .map( action -> String.format( "  %s", buildActionTag( action, objectMapper ) ) )
            .collect( Collectors.joining( "\n" ) );
    }

    @SneakyThrows
    private String buildActionTag( Action action, ObjectMapper objectMapper )
    {
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append( String.format(
            "<action tool=\"%s\" status=\"%s\">",
            action.getToolName(), action.getState().name()
        ) );
        if ( action.getResult() != null )
        {
            stringBuilder.append( String.format(
                "  <result>%s</result>",
                action.getResult().getData() instanceof String
                    ? action.getResult().getData()
                    : objectMapper.writeValueAsString( action.getResult().getData() )
            ) );
        }

        stringBuilder.append( "<action>" );
        return stringBuilder.toString();
    }
}
