package com.suncode.pwfl.assistant;

import com.suncode.pwfl.assistant.agent.Agent;
import com.suncode.pwfl.assistant.model.Action;
import com.suncode.pwfl.assistant.model.AgentActionState;
import com.suncode.pwfl.assistant.model.AgentConfig;
import com.suncode.pwfl.assistant.model.AgentState;
import com.suncode.pwfl.assistant.model.AgentTaskState;
import com.suncode.pwfl.assistant.model.Task;
import com.suncode.pwfl.assistant.phase.AgentActionPlanningPhase;
import com.suncode.pwfl.assistant.phase.AgentActionUsagePhase;
import com.suncode.pwfl.assistant.phase.AgentTaskPlanningPhase;
import com.suncode.pwfl.assistant.phase.AgentThinkingPhase;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;

import java.util.Optional;

@Slf4j
public class AgentExecutor
{

    private final AgentConfig config;

    private final AgentThinkingPhase thinkingPhase;
    private final AgentTaskPlanningPhase taskPlanningPhase;
    private final AgentActionPlanningPhase actionPlanningPhase;
    private final AgentActionUsagePhase actionPhase;
    
    public AgentExecutor( AgentContext context )
    {
        this.config = context.getConfig();

        this.thinkingPhase = new AgentThinkingPhase( context );
        this.taskPlanningPhase = new AgentTaskPlanningPhase( context );
        this.actionPlanningPhase = new AgentActionPlanningPhase( context );
        this.actionPhase = new AgentActionUsagePhase( context );
    }

    @SneakyThrows
    public void execute( Agent agent, String userMessage)
    {
        log.debug( "[{}] Executing agent", agent );

        if ( agent.getThinkingPrompt().isPresent() )
        {
            log.debug( "[{}] Executing thinking phase", agent );
            this.thinkingPhase.execute( agent, agent.getThinkingPrompt().get(), userMessage );
        }

        AgentState state = agent.getState();
        for ( int step = 0; step < config.getMaxSteps(); step++ )
        {
            state.setStep( 0 );

            log.info( "[{}][{}] Executing task planning phase", agent, step );
            this.taskPlanningPhase.execute( agent, userMessage );

            log.debug( "[{}][{}] Executing action planning phase", agent, step );
            this.actionPlanningPhase.execute( agent, userMessage );

            log.info( "[{}][{}] Executing action usage phase", agent, step );
            this.actionPhase.execute( agent, userMessage );

            Optional<Task> currentTaskOptional = state.getTasks().stream()
                .filter( task -> task.getUuid().equals( state.getTask() ) )
                .findFirst();
            if( currentTaskOptional.isEmpty() )
            {
                log.warn( "[{}][{}] No task found", agent, step );
                break;
            }

            Task currentTask = currentTaskOptional.get();
            Optional<Action> currentActionOptional = currentTask.getActions().stream()
                .filter( action -> action.getUuid().equals( state.getAction() ) )
                .findFirst();

            // inaczej niż stringiem wyznaczać "final_answer" żeby agentExecutor nie musiał znać konkretnych tooli
            // parametryzować per agent, najlepiej żeby tool provider miał w interfejsie metodę która to zwróci i każdy musi to zaimplementować (albo
            // default impl w interfejsie zwraca final_answer)
            if ( currentActionOptional.isEmpty())
            {
                log.warn( "[{}][{}] No action found", agent, step );
                break;
            }

            // TODO zrefaktorować, by nie polegać na konkretnych nazwach tool'i
            Action currentAction = currentActionOptional.get();
            if ( currentAction.getToolName().equals( "answer_to_agent" ) || currentAction.getToolName().equals( "final_answer" ))
            {
                log.info( "[{}][{}] Setting final answer", agent, step );
                state.setFinalAnswer( currentActionOptional.get().getResult() );
                break;
            }

            if ( currentAction.getState() == AgentActionState.COMPLETED )
            {
                log.info( "[{}][{}] Setting task completed", agent, step );
                currentTask.setState( AgentTaskState.COMPLETED );
            }
            else if ( currentAction.getState() == AgentActionState.FAILED )
            {
                currentTask.incrementFailureCount();
                log.info( "[{}][{}] Increasing failure count to {}", agent, step, currentTask.getFailureCount() );
                if ( currentTask.getFailureCount() > 2 )
                {
                    log.info( "[{}][{}] Setting task skipped", agent, step );
                    currentTask.setState( AgentTaskState.SKIPPED );
                }
                else
                {
                    log.info( "[{}][{}] Setting task failed", agent, step );
                    currentTask.setState( AgentTaskState.FAILED );
                }
            }
        }

    }


}
