package com.suncode.pwfl.xpdl.builder;

import com.suncode.pwfl.workflow.process.map.Activity;
import com.suncode.pwfl.workflow.process.map.Package;
import com.suncode.pwfl.workflow.process.map.Process;
import com.suncode.pwfl.workflow.process.map.Variable;
import com.suncode.pwfl.workflow.process.map.VariableRef;
import com.suncode.pwfl.workflow.process.map.VariableType;
import com.suncode.pwfl.workflow.process.map.element.ActivityElement;
import com.suncode.pwfl.workflow.process.map.transition.Gateway;
import com.suncode.pwfl.workflow.process.map.transition.Transition;
import com.suncode.pwfl.workflow.process.map.zipped.ZippedActivity;
import com.suncode.pwfl.workflow.process.map.zipped.ZippedPackage;
import com.suncode.pwfl.workflow.process.map.zipped.ZippedProcess;
import lombok.AccessLevel;
import lombok.NoArgsConstructor;
import lombok.Setter;

import java.time.LocalDateTime;
import java.util.Comparator;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.function.Supplier;
import java.util.stream.Collectors;

@NoArgsConstructor( access = AccessLevel.PACKAGE )
public class XpdlBuilderService
{

    private static final int ACTIVITY_BLOCK_WIDTH = 90;

    private static final int ACTIVITY_BLOCK_HEIGHT = 60;

    private static final int ACTIVITY_BLOCK_MARGIN = 50;

    private static final int WORKFLOW_POINT_SIZE = 40;

    private static final int PARTICIPANT_LABEL_OFFSET = 30;

    private static final int MINIMUM_MARGIN_FOR_PROCESS_START =
        PARTICIPANT_LABEL_OFFSET + ACTIVITY_BLOCK_MARGIN + WORKFLOW_POINT_SIZE + ACTIVITY_BLOCK_MARGIN;

    @Setter( AccessLevel.PACKAGE )
    private Supplier<String> attachmentDirectoryIdGenerator = () -> UUID.randomUUID().toString();

    public static String buildBy( ZippedPackage pkg )
    {
        XpdlBuilderService xpdlBuilderService = new XpdlBuilderService();
        return xpdlBuilderService.buildBy( pkg, LocalDateTime.now() );
    }

    String buildBy( ZippedPackage zippedPackage, LocalDateTime createdTimestamp )
    {
        Package pkg = zippedPackage.getModel();

        XpdlBuilder xpdlBuilder = new XpdlBuilder()
            .withPackageId( pkg.getId() )
            .withPackageName( pkg.getName() )
            .withAuthor( "Suncode" )
            .withCreated( createdTimestamp );

        for ( ZippedProcess zippedProcess : zippedPackage.getZippedProcesses() )
        {
            appendProcess( createdTimestamp, zippedProcess, xpdlBuilder );
        }

        return xpdlBuilder.build();
    }

    private void appendProcess( LocalDateTime createdTimestamp,
                                ZippedProcess zippedProcess,
                                XpdlBuilder xpdlBuilder )
    {
        offsetElementsForProcessStartIfNecessary( zippedProcess );

        Process process = zippedProcess.getModel();

        XpdlProcessBuilder processBuilder = XpdlProcessBuilder
            .create( process.getId(), process.getName() )
            .withCreated( createdTimestamp )
            .withAttachmentDirectory( attachmentDirectoryIdGenerator.get() )
            .addParticipant(
                XpdlParticipantBuilder.create( "uczestnik_1", "Uczestnik 1" )
            );
        xpdlBuilder.addProcess( processBuilder );

        appendVariables( createdTimestamp, zippedProcess, processBuilder );
        appendTransitions( zippedProcess, processBuilder );
        appendActivities( zippedProcess, processBuilder );

        appendProcessEnds( zippedProcess, processBuilder );
    }

    private void offsetElementsForProcessStartIfNecessary( ZippedProcess zippedProcess )
    {
        ZippedActivity startingZippedActivity = getStartingZippedActivity( zippedProcess );

        int startingActivityX = startingZippedActivity.getElement().getX();
        int estimatedProcessStartPositionX = startingActivityX - MINIMUM_MARGIN_FOR_PROCESS_START;

        if ( estimatedProcessStartPositionX >= 0 )
        {
            return;
        }

        int offsetX = -estimatedProcessStartPositionX;
        for ( ZippedActivity zippedActivity : zippedProcess.getZippedActivities() )
        {
            ActivityElement activityElement = zippedActivity.getElement();
            activityElement.setX( offsetX + activityElement.getX() );
        }
    }

    private void appendVariables( LocalDateTime createdTimestamp, ZippedProcess zippedProcess,
                                  XpdlProcessBuilder processBuilder )
    {
        for ( Variable variable : zippedProcess.getModel().getVariables() )
        {
            VariableType type = variable.getType();
            XpdlVariableBuilder variableBuilder = switch ( type )
            {
                case STRING -> XpdlVariableBuilder.createString( variable.getId(), variable.getName() );
                case INTEGER -> XpdlVariableBuilder.createInteger( variable.getId(), variable.getName() );
                case FLOAT -> XpdlVariableBuilder.createFloat( variable.getId(), variable.getName() );
                case BOOLEAN -> XpdlVariableBuilder.createBoolean( variable.getId(), variable.getName() );
                case DATE -> XpdlVariableBuilder.createDate( variable.getId(), variable.getName() );
                case DATE_TIME -> XpdlVariableBuilder.createDateTime( variable.getId(), variable.getName() );
            };

            variableBuilder = variableBuilder
                .withCreationDate( createdTimestamp )
                .withModificationDate( createdTimestamp );

            processBuilder.addVariable( variableBuilder );
        }
    }

    private void appendTransitions( ZippedProcess zippedProcess, XpdlProcessBuilder processBuilder )
    {
        Set<String> allGatewayIds = getAllGatewayIds( zippedProcess );

        int newTransitionId = 1;
        for ( Transition transition : zippedProcess.getModel().getTransitions() )
        {
            String sourceId = transition.getSourceId();
            String targetId = transition.getTargetId();

            // flatMap gateway transitions into intermediate transitions
            // otherwise, generate direct transitions
            if ( allGatewayIds.contains( targetId ) )
            {
                List<Transition> gatewayTransitions = zippedProcess.getModel().getTransitions().stream()
                    .filter( innerTransition -> innerTransition.getSourceId().equals( targetId ) )
                    .toList();

                for ( Transition gatewayTransition : gatewayTransitions )
                {
                    String targetActivityId = gatewayTransition.getTargetId();

                    XpdlTransitionBuilder flatTransitionBuilder = XpdlTransitionBuilder
                        .create(
                            "transition-" + newTransitionId,
                            sourceId, targetActivityId
                        )
                        .withConditionText( gatewayTransition.getConditionText() );
                    newTransitionId += 1;

                    processBuilder.addTransition( flatTransitionBuilder );
                }
            }
            else if ( !allGatewayIds.contains( sourceId ) )
            {
                XpdlTransitionBuilder flatTransitionBuilder = XpdlTransitionBuilder.create(
                    "transition-" + newTransitionId,
                    sourceId, targetId
                );
                newTransitionId += 1;

                processBuilder.addTransition( flatTransitionBuilder );
            }
        }
    }

    private void appendActivities( ZippedProcess zippedProcess, XpdlProcessBuilder processBuilder )
    {
        for ( ZippedActivity zippedActivity : zippedProcess.getZippedActivities() )
        {
            Activity activityModel = zippedActivity.getModel();
            ActivityElement activityElement = zippedActivity.getElement();

            XpdlActivityBuilder activityBuilder = XpdlActivityBuilder
                .create( activityModel.getId(), activityModel.getName(), "uczestnik_1",
                         activityElement.getX(), activityElement.getY() )
                .withVariables(
                    activityModel.getVariableRefs().stream()
                        .sorted( Comparator.comparing( VariableRef::getPosition ) )
                        .map( variableRef -> XpdlVariableRefBuilder.create( variableRef.getId(), variableRef.getType() ) )
                        .toList()
                );

            appendAcceptButtons( processBuilder, zippedActivity, activityBuilder );
            appendTransitionRestrictions( zippedProcess, processBuilder, zippedActivity, activityBuilder );

            processBuilder.addActivity( activityBuilder );
        }
    }

    private void appendAcceptButtons( XpdlProcessBuilder processBuilder, ZippedActivity zippedActivity,
                                      XpdlActivityBuilder activityBuilder )
    {
        Activity activityModel = zippedActivity.getModel();

        List<XpdlAcceptButtonBuilder> acceptButtonBuilders = activityModel.getAcceptButtons().stream()
            .map( acceptButton -> {
                String targetActivityId = acceptButton.getTargetActivityId();

                String transitionId = processBuilder.getTransitions().stream()
                    .filter( xpdlTransitionBuilder -> {
                        return xpdlTransitionBuilder.getToActivityId().equals( targetActivityId )
                            && xpdlTransitionBuilder.getFromActivityId().equals( activityModel.getId() );
                    } )
                    .map( XpdlTransitionBuilder::getId )
                    .findFirst()
                    .orElseThrow();

                return XpdlAcceptButtonBuilder.create(
                    acceptButton.getId(), acceptButton.getName(), transitionId
                );
            } )
            .toList();

        activityBuilder.withAcceptButtons( acceptButtonBuilders );
    }

    private void appendTransitionRestrictions( ZippedProcess zippedProcess, XpdlProcessBuilder processBuilder,
                                               ZippedActivity zippedActivity, XpdlActivityBuilder activityBuilder )
    {
        Activity activityModel = zippedActivity.getModel();

        List<XpdlTransitionBuilder> incomingFlatTransitions = processBuilder.getTransitions().stream()
            .filter( transition -> transition.getToActivityId().equals( activityModel.getId() ) )
            .toList();

        List<XpdlTransitionBuilder> outcomingFlatTransitions = processBuilder.getTransitions().stream()
            .filter( transition -> transition.getFromActivityId().equals( activityModel.getId() ) )
            .toList();

        if ( incomingFlatTransitions.size() < 2 && outcomingFlatTransitions.size() < 2 )
        {
            return;
        }

        XpdlTransitionRestrictions transitionRestrictions = XpdlTransitionRestrictions.create();

        if ( incomingFlatTransitions.size() >= 2 )
        {
            transitionRestrictions.withJoinType( activityModel.getJoinType() );
        }

        if ( outcomingFlatTransitions.size() >= 2 )
        {
            zippedProcess.getModel().getGateways().stream()
                .filter( gateway -> zippedProcess.getModel().getTransitions().stream()
                    .anyMatch( transition -> {
                        return transition.getSourceId().equals( activityModel.getId() )
                            && transition.getTargetId().equals( gateway.getId() );
                    } )
                )
                .findFirst()
                .ifPresent( gateway -> {
                    transitionRestrictions
                        .withSplitType( switch ( gateway.getGatewayType() )
                                        {
                                            case AND -> XpdlSplitType.AND;
                                            case XOR -> XpdlSplitType.XOR;
                                        } )
                        .withTransitionRefs(
                            outcomingFlatTransitions.stream()
                                .map( XpdlTransitionBuilder::getId )
                                .toList()
                        );
                } );
        }

        activityBuilder.withTransitionRestrictions( transitionRestrictions );
    }

    private void appendProcessEnds( ZippedProcess zippedProcess, XpdlProcessBuilder processBuilder )
    {
        ZippedActivity zippedStartingActivity = getStartingZippedActivity( zippedProcess );
        List<ZippedActivity> endingZippedActivities = getEndingZippedActivities( zippedProcess );

        int processStartingPositionX = zippedStartingActivity.getElement().getX()
            - ACTIVITY_BLOCK_MARGIN - WORKFLOW_POINT_SIZE;
        int processStartingPositionY = zippedStartingActivity.getElement().getY()
            + ((ACTIVITY_BLOCK_HEIGHT - WORKFLOW_POINT_SIZE) / 2);

        processBuilder.addWorkflowStart( XpdlWorkflowPointBuilder.create(
            "uczestnik_1", zippedStartingActivity.getModel().getId(),
            processStartingPositionX, processStartingPositionY
        ) );

        for ( ZippedActivity endingZippedActivity : endingZippedActivities )
        {
            int processEndingPositionX = endingZippedActivity.getElement().getX()
                + ACTIVITY_BLOCK_WIDTH + ACTIVITY_BLOCK_MARGIN;
            int processEndingPositionY = endingZippedActivity.getElement().getY()
                + ((ACTIVITY_BLOCK_HEIGHT - WORKFLOW_POINT_SIZE) / 2);

            processBuilder.addWorkflowEnd( XpdlWorkflowPointBuilder.create(
                "uczestnik_1", endingZippedActivity.getModel().getId(),
                processEndingPositionX,
                processEndingPositionY
            ) );
        }
    }

    private ZippedActivity getStartingZippedActivity( ZippedProcess zippedProcess )
    {
        Set<String> transitionEndingActivityIds = zippedProcess.getModel().getTransitions().stream()
            .map( Transition::getTargetId )
            .collect( Collectors.toSet() );

        return zippedProcess.getZippedActivities().stream()
            .filter( zippedActivity -> {
                return !transitionEndingActivityIds.contains( zippedActivity.getModel().getId() );
            } )
            .findFirst()
            .orElseThrow();
    }

    private List<ZippedActivity> getEndingZippedActivities( ZippedProcess zippedProcess )
    {
        Set<String> transitionStartingActivityIds = zippedProcess.getModel().getTransitions().stream()
            .map( Transition::getSourceId )
            .collect( Collectors.toSet() );

        return zippedProcess.getZippedActivities().stream()
            .filter( zippedActivity -> {
                return !transitionStartingActivityIds.contains( zippedActivity.getModel().getId() );
            } )
            .toList();
    }

    private Set<String> getAllGatewayIds( ZippedProcess zippedProcess )
    {
        return zippedProcess.getModel().getGateways().stream()
            .map( Gateway::getId )
            .collect( Collectors.toSet() );
    }

}
