package com.suncode.pwfl.xpdl.builder;

import lombok.Getter;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import java.util.ArrayList;
import java.util.List;

@Getter
class XpdlActivityBuilder
    extends XpdlBaseBuilder
{

    private final String id;

    private final String name;

    private final String participantId;

    private final int xOffset;

    private final int yOffset;

    private List<XpdlVariableRefBuilder> variables = new ArrayList<>();

    private List<XpdlAcceptButtonBuilder> acceptButtons = new ArrayList<>();

    private XpdlTransitionRestrictions transitionRestrictions;

    private XpdlActivityBuilder( String id, String name, String participantId, int xOffset, int yOffset )
    {
        this.id = id;
        this.name = name;
        this.participantId = participantId;
        this.xOffset = xOffset;
        this.yOffset = yOffset;
    }

    public static XpdlActivityBuilder create( String id, String name, String participantId, int xOffset, int yOffset )
    {
        return new XpdlActivityBuilder( id, name, participantId, xOffset, yOffset );
    }

    public XpdlActivityBuilder withVariables( List<XpdlVariableRefBuilder> variables )
    {
        this.variables = new ArrayList<>( variables );
        return this;
    }

    public XpdlActivityBuilder withAcceptButtons( List<XpdlAcceptButtonBuilder> acceptButtons )
    {
        this.acceptButtons = new ArrayList<>( acceptButtons );
        return this;
    }

    public XpdlActivityBuilder withTransitionRestrictions( XpdlTransitionRestrictions transitionRestrictions )
    {
        this.transitionRestrictions = transitionRestrictions;
        return this;
    }

    @Override
    public Element buildElement( Document document )
    {
        Element activity = document.createElementNS( XPDL_NAMESPACE, ELEMENT_ACTIVITY );
        activity.setAttribute( ATTR_ID, id );
        activity.setAttribute( ATTR_NAME, name );

        Element implementation = document.createElementNS( XPDL_NAMESPACE, ELEMENT_IMPLEMENTATION );
        Element no = document.createElementNS( XPDL_NAMESPACE, ELEMENT_NO );
        implementation.appendChild( no );
        activity.appendChild( implementation );

        Element performer = document.createElementNS( XPDL_NAMESPACE, ELEMENT_PERFORMER );
        performer.setTextContent( participantId );
        activity.appendChild( performer );

        if ( transitionRestrictions != null )
        {
            Element transitionRestrictionsElement = document
                .createElementNS( XPDL_NAMESPACE, "xpdl:TransitionRestrictions" );
            activity.appendChild( transitionRestrictionsElement );

            Element transitionRestrictionElement = document
                .createElementNS( XPDL_NAMESPACE, "xpdl:TransitionRestriction" );
            transitionRestrictionsElement.appendChild( transitionRestrictionElement );

            if ( transitionRestrictions.getJoinType() != null )
            {
                Element joinTypeElement = document.createElementNS( XPDL_NAMESPACE, "xpdl:Join" );
                joinTypeElement.setAttribute( "Type", transitionRestrictions.getJoinType().name() );

                transitionRestrictionElement.appendChild( joinTypeElement );
            }

            if ( transitionRestrictions.getSplitType() != null )
            {
                Element splitTypeElement = document.createElementNS( XPDL_NAMESPACE, "xpdl:Split" );
                splitTypeElement.setAttribute( "Type", transitionRestrictions.getSplitType().name() );

                transitionRestrictionElement.appendChild( splitTypeElement );

                if ( transitionRestrictions.getTransitionRefs() != null )
                {
                    Element transitionRefsElement = document.createElementNS( XPDL_NAMESPACE, "xpdl:TransitionRefs" );
                    splitTypeElement.appendChild( transitionRefsElement );

                    for ( String transitionRef : transitionRestrictions.getTransitionRefs() )
                    {
                        Element transitionRefElement = document.createElementNS( XPDL_NAMESPACE, "xpdl:TransitionRef" );
                        transitionRefElement.setAttribute( "Id", transitionRef );

                        transitionRefsElement.appendChild( transitionRefElement );
                    }
                }
            }
        }

        Element extendedAttributes = document.createElementNS( XPDL_NAMESPACE, ELEMENT_EXTENDED_ATTRIBUTES );

        addExtendedAttribute( document, extendedAttributes, EXT_ATTR_JAWE_GRAPH_PARTICIPANT_ID, participantId );
        addExtendedAttribute( document, extendedAttributes, EXT_ATTR_JAWE_GRAPH_OFFSET, xOffset + "," + yOffset );

        for ( XpdlVariableRefBuilder variable : variables )
        {
            String extAttrVariableRefType = switch (variable.getType()) {
                case UPDATE -> EXT_ATTR_VARIABLE_TO_PROCESS_UPDATE;
                case VIEW -> EXT_ATTR_VARIABLE_TO_PROCESS_VIEW;
                case HIDDEN -> EXT_ATTR_VARIABLE_TO_PROCESS_HIDDEN;
            };

            addExtendedAttribute( document, extendedAttributes, extAttrVariableRefType, variable.getId() );
        }

        for ( XpdlAcceptButtonBuilder acceptButton : acceptButtons )
        {
            String actionAcceptButtonString = String.format(
                "actionName=%s\n"
                    + "buttonName=%s\n"
                    + "action1=Action=%s\n"
                    + "checkForm=FALSE\n"
                    + "saveBeforeAccept=FALSE\n"
                    + "forwardTo=%s\n"
                    + "filled=false\n"
                    + "hideFrame=false",
                acceptButton.getId(),
                acceptButton.getName(),
                acceptButton.getId(),
                acceptButton.getForwardTo()
            );

            addExtendedAttribute( document, extendedAttributes, "ACTION_ACCEPT_BUTTON", actionAcceptButtonString );
        }

        addExtendedAttribute( document, extendedAttributes,
                              EXT_ATTR_EVENTS,
                              "{\"FORM\":{},\"VARIABLE\":{},\"GRID\":{},\"DT_BUTTON\":{},\"ACTION_BUTTON\":{},\"ACCEPT_BUTTON\":{}}" );

        activity.appendChild( extendedAttributes );

        return activity;
    }
} 