package com.suncode.pwfl.workflow.process.map.builder;

import com.suncode.pwfl.workflow.process.map.Activity;
import com.suncode.pwfl.workflow.process.map.Package;
import com.suncode.pwfl.workflow.process.map.Process;
import com.suncode.pwfl.workflow.process.map.builder.function.FunctionCall;
import com.suncode.pwfl.workflow.process.map.builder.function.FunctionCallArgument;
import com.suncode.pwfl.workflow.process.map.Variable;
import com.suncode.pwfl.workflow.process.map.VariableType;
import com.suncode.pwfl.workflow.process.map.transition.Transition;

import java.text.Normalizer;
import java.util.Map;
import java.util.UUID;

class ProcessMapBuilderImpl
    implements ProcessMapBuilder
{
    private Process process;

    private static final String REMOVE_ACCENTS_REGEX = "\\p{M}";

    ProcessMapBuilderImpl()
    {
    }

    @Override
    public void setProcessId( String id )
    {
        getProcess().setId( id );
    }

    @Override
    public void setProcessName( String name )
    {
        getProcess().setName( name );
    }

    @Override
    public void createActivity( String id, String name )
    {
        getProcess().getActivities().add( new Activity( id, name ) );
    }

    @Override
    public void createDirectTransition( String activityIdFrom, String activityIdTo )
    {
        getProcess().getTransitions().add( new Transition(
            UUID.randomUUID().toString(),
            activityIdFrom, activityIdTo
        ) );
    }

    @Override
    public void createVariable( String id, String name, VariableType type )
    {
        getProcess().getVariables().add( new Variable( type, id, name ) );
    }

    @Override
    public Package build()
    {
        Package pack = new Package();
        pack.getProcesses().clear();
        pack.getProcesses().add( getProcess() );
        return pack;
    }

    Process getProcess()
    {
        if ( process == null )
        {
            this.process = new Process();
        }

        return this.process;
    }

    void apply( FunctionCall functionCall )
    {
        final Map<String, FunctionCallArgument> arguments = functionCall.arguments();
        switch ( functionCall.functionName() )
        {
            case "set_process_id" -> setProcessId(
                normalizeAndRemoveAccents( arguments.get( "id" ).value() )
            );
            case "set_process_name" -> setProcessName(
                normalizeAndRemoveAccents( arguments.get( "name" ).value() )
            );
            case "create_activity" -> createActivity(
                normalizeAndRemoveAccents( arguments.get( "id" ).value() ),
                arguments.get( "name" ).value()
            );
            case "create_transition" -> createDirectTransition(
                normalizeAndRemoveAccents( arguments.get( "activityIdFrom" ).value() ),
                normalizeAndRemoveAccents( arguments.get( "activityIdTo" ).value() )
            );
            case "create_variable" -> createVariable(
                normalizeAndRemoveAccents(
                    arguments.get( "id" ).value() ), arguments.get( "name" ).value(),
                VariableType.valueOf( arguments.get( "type" ).value() )
            );
        }
    }

    private String normalizeAndRemoveAccents( String input )
    {
        return this.removeAccents( this.normalize( input ) );
    }

    private String normalize( String input )
    {
        return Normalizer.normalize( input, Normalizer.Form.NFKD );
    }

    private String removeAccents( String input )
    {
        return input.replaceAll( REMOVE_ACCENTS_REGEX, "" );
    }
}
