package com.suncode.pwfl.xpdl.builder;

import lombok.Getter;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import java.time.LocalDateTime;

@Getter
class XpdlVariableBuilder
    extends XpdlBaseBuilder
{

    @Getter
    public enum VariableType
    {
        INTEGER( STRING_TYPE_INTEGER ),
        STRING( STRING_TYPE_STRING ),
        DATETIME( STRING_TYPE_DATETIME ),
        BOOLEAN( STRING_TYPE_BOOLEAN ),
        FLOAT( STRING_TYPE_FLOAT );

        private final String type;

        VariableType( String type )
        {
            this.type = type;
        }

    }

    @Getter
    public enum StringSubtype
    {
        NONE( "" ),
        DATE( "DATE" );

        private final String subtype;

        StringSubtype( String subtype )
        {
            this.subtype = subtype;
        }

    }

    private final String id;

    private final String name;

    private final VariableType type;

    private StringSubtype subtype = StringSubtype.NONE;

    private String format;

    private String decimalPrecision;

    private String thousandSeparator;

    private LocalDateTime creationDate = LocalDateTime.now();

    private LocalDateTime modificationDate = LocalDateTime.now();

    private XpdlVariableBuilder( String id, String name, VariableType type )
    {
        this.id = id;
        this.name = name;
        this.type = type;
    }

    private XpdlVariableBuilder withSubtype( StringSubtype subtype )
    {
        this.subtype = subtype;
        return this;
    }

    private XpdlVariableBuilder withFormat( String format )
    {
        this.format = format;
        return this;
    }

    private XpdlVariableBuilder withDecimalPrecision( String decimalPrecision )
    {
        this.decimalPrecision = decimalPrecision;
        return this;
    }

    private XpdlVariableBuilder withThousandSeparator( String thousandSeparator )
    {
        this.thousandSeparator = thousandSeparator;
        return this;
    }

    public XpdlVariableBuilder withCreationDate( LocalDateTime creationDate )
    {
        this.creationDate = creationDate;
        return this;
    }

    public XpdlVariableBuilder withCreationDate( String creationDate )
    {
        this.creationDate = LocalDateTime.parse( creationDate, DATE_TIME_FORMATTER );
        return this;
    }

    public XpdlVariableBuilder withModificationDate( LocalDateTime modificationDate )
    {
        this.modificationDate = modificationDate;
        return this;
    }

    public XpdlVariableBuilder withModificationDate( String modificationDate )
    {
        this.modificationDate = LocalDateTime.parse( modificationDate, DATE_TIME_FORMATTER );
        return this;
    }

    public static XpdlVariableBuilder createInteger( String id, String name )
    {
        return new XpdlVariableBuilder( id, name, VariableType.INTEGER )
            .withFormat( "0,000" )
            .withThousandSeparator( " " );
    }

    public static XpdlVariableBuilder createFloat( String id, String name )
    {
        return new XpdlVariableBuilder( id, name, VariableType.FLOAT )
            .withFormat( "0,000.00" )
            .withDecimalPrecision( "2" )
            .withThousandSeparator( " " );
    }

    public static XpdlVariableBuilder createString( String id, String name )
    {
        return new XpdlVariableBuilder( id, name, VariableType.STRING );
    }

    public static XpdlVariableBuilder createDate( String id, String name )
    {
        return new XpdlVariableBuilder( id, name, VariableType.STRING )
            .withSubtype( StringSubtype.DATE )
            .withFormat( "Y-m-d" );
    }

    public static XpdlVariableBuilder createDateTime( String id, String name )
    {
        return new XpdlVariableBuilder( id, name, VariableType.DATETIME );
    }

    public static XpdlVariableBuilder createBoolean( String id, String name )
    {
        return new XpdlVariableBuilder( id, name, VariableType.BOOLEAN );
    }

    @Override
    public Element buildElement( Document document )
    {
        Element dataField = document.createElementNS( XPDL_NAMESPACE, ELEMENT_DATA_FIELD );
        dataField.setAttribute( ATTR_ID, id );
        dataField.setAttribute( ATTR_NAME, name );
        dataField.setAttribute( ATTR_IS_ARRAY, STRING_FALSE );

        Element dataType = document.createElementNS( XPDL_NAMESPACE, ELEMENT_DATA_TYPE );
        Element basicType = document.createElementNS( XPDL_NAMESPACE, ELEMENT_BASIC_TYPE );
        basicType.setAttribute( ATTR_TYPE, type.getType() );
        dataType.appendChild( basicType );
        dataField.appendChild( dataType );

        Element extendedAttributes = document.createElementNS( XPDL_NAMESPACE, ELEMENT_EXTENDED_ATTRIBUTES );

        if ( subtype == StringSubtype.DATE )
        {
            addExtendedAttribute( document, extendedAttributes, EXT_ATTR_SUBTYPE, subtype.getSubtype() );
        }

        addExtendedAttribute( document, extendedAttributes, EXT_ATTR_ARRAY, STRING_FALSE );

        if ( format != null )
        {
            addExtendedAttribute( document, extendedAttributes, EXT_ATTR_FORMAT, format );
        }

        if ( decimalPrecision != null )
        {
            addExtendedAttribute( document, extendedAttributes, EXT_ATTR_DECIMAL_PRECISION, decimalPrecision );
        }

        if ( thousandSeparator != null )
        {
            addExtendedAttribute( document, extendedAttributes, EXT_ATTR_THOUSAND_SEPARATOR, thousandSeparator );
        }

        addExtendedAttribute( document, extendedAttributes, EXT_ATTR_CREATION_DATE, formatDateTime( creationDate ) );
        addExtendedAttribute( document, extendedAttributes, EXT_ATTR_MODIFICATION_DATE, formatDateTime( modificationDate ) );

        dataField.appendChild( extendedAttributes );

        return dataField;
    }
} 