package com.suncode.pwfl.workflow.process.map.builder;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.suncode.pwfl.prompting.LlmService;
import com.suncode.pwfl.workflow.process.map.Process;
import com.suncode.pwfl.workflow.process.map.Variable;
import com.suncode.pwfl.workflow.process.map.VariableRef;
import com.suncode.pwfl.workflow.process.map.builder.intention.UnknownIntentionException;
import com.suncode.pwfl.workflow.process.map.builder.intention.UserIntention;
import com.suncode.pwfl.workflow.process.map.builder.intention.UserIntentionEvaluation;
import lombok.extern.slf4j.Slf4j;

import java.util.List;
import java.util.stream.IntStream;

@Slf4j
public class LlmProcessMapBuilderImpl
    implements LlmProcessMapBuilder
{
    private final LlmService llmService;

    private final LlmProcessMapBuilderInstructionProvider instructionProvider = new LlmProcessMapBuilderInstructionProvider();

    private final ObjectMapper objectMapper = new ObjectMapper();

    public LlmProcessMapBuilderImpl( LlmService llmService )
    {
        this.llmService = llmService;
    }

    @Override
    public Process buildProcessMapFromPrompt( String prompt )
    {
        var userIntention = resolveUserIntention( prompt );
        if ( userIntention == UserIntention.UNKNOWN_ACTION )
        {
            throw new UnknownIntentionException();
        }
        String finalPrompt = getEnhancedPrompt( userIntention, prompt );
        return buildProcessMapFromFinalPrompt( finalPrompt );
    }

    private UserIntention resolveUserIntention( String prompt )
    {
        try
        {
            String chatResponse = llmService.chat( instructionProvider.resolveUserIntentionInstruction(), prompt );
            var userIntentionEvaluation = objectMapper.readValue( chatResponse, UserIntentionEvaluation.class );
            return UserIntention.valueOf( userIntentionEvaluation.action() );
        }
        catch ( Exception e )
        {
            log.info( "Couldn't resolve user intention from user prompt", e );
            return UserIntention.UNKNOWN_ACTION;
        }
    }

    private String getEnhancedPrompt( UserIntention userIntention, String prompt )
    {
        if ( userIntention == UserIntention.CREATE_PROCESS_WITH_DETAILS )
        {
            return prompt;
        }

        try
        {
            return llmService.chat( instructionProvider.enhanceUserPromptInstruction(), prompt );
        }
        catch ( Exception e )
        {
            log.info( "Couldn't enhance user prompt", e );
            return prompt;
        }
    }

    private Process buildProcessMapFromFinalPrompt( String prompt )
    {
        try
        {
            String chatResponse = llmService.chat( instructionProvider.processBuildInstruction(), prompt );
            var processMapBuilderResponse = objectMapper.readValue( chatResponse, LlmProcessMapBuilderResponse.class );
            var processMapBuilder = new ProcessMapBuilderImpl();
            processMapBuilderResponse.functionCalls().stream()
                .forEach( processMapBuilder::apply );
            addVariableRefsToActivities( processMapBuilder );
            return processMapBuilder.getProcess();
        }
        catch ( Exception e )
        {
            log.info( "Couldn't build process map from user prompt", e );
            return new Process();
        }
    }

    private void addVariableRefsToActivities( ProcessMapBuilderImpl processMapBuilder )
    {
        List<VariableRef> variableRefs = IntStream.range( 0, processMapBuilder.getProcess().getVariables().size() )
            .mapToObj( i -> {
                Variable variable = processMapBuilder.getProcess().getVariables().get( i );
                return new VariableRef( variable.getId(), true, true, i );
            } )
            .toList();

        for ( var activity : processMapBuilder.getProcess().getActivities() )
        {
            activity.getVariableRefs().addAll( variableRefs );
        }
    }
}
