package com.suncode.pwfl.xpdl.builder;

import lombok.Getter;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import java.util.ArrayList;
import java.util.List;

@Getter
class XpdlTransitionBuilder
    extends XpdlBaseBuilder
{

    private final String id;

    private final String fromActivityId;

    private final String toActivityId;

    private XpdlTransitionStyle transitionStyle = XpdlTransitionStyle.NO_ROUTING_SPLINE;

    private XpdlFirstLineDirection firstLineDirection = XpdlFirstLineDirection.RIGHT;

    private List<XpdlLinePathBuilder> linePaths = new ArrayList<>();

    private XpdlTransitionBuilder( String id, String fromActivityId, String toActivityId )
    {
        this.id = id;
        this.fromActivityId = fromActivityId;
        this.toActivityId = toActivityId;
    }

    public static XpdlTransitionBuilder create( String id, String fromActivityId, String toActivityId )
    {
        return new XpdlTransitionBuilder( id, fromActivityId, toActivityId );
    }

    public XpdlTransitionBuilder withTransitionStyle( XpdlTransitionStyle transitionStyle )
    {
        this.transitionStyle = transitionStyle;
        return this;
    }

    public XpdlTransitionBuilder withFirstLineDirection( XpdlFirstLineDirection firstLineDirection )
    {
        this.firstLineDirection = firstLineDirection;
        return this;
    }

    public XpdlTransitionBuilder addLinePath( XpdlLinePathBuilder linePath )
    {
        this.linePaths.add( linePath );
        return this;
    }

    public XpdlTransitionBuilder withLinePaths( List<XpdlLinePathBuilder> linePaths )
    {
        this.linePaths = linePaths;
        return this;
    }

    @Override
    public Element buildElement( Document document )
    {
        Element transition = document.createElementNS( XPDL_NAMESPACE, ELEMENT_TRANSITION );
        transition.setAttribute( ATTR_ID, id );
        transition.setAttribute( ATTR_FROM, fromActivityId );
        transition.setAttribute( ATTR_TO, toActivityId );

        Element extendedAttributes = document.createElementNS( XPDL_NAMESPACE, ELEMENT_EXTENDED_ATTRIBUTES );

        addExtendedAttribute( document, extendedAttributes, EXT_ATTR_JAWE_GRAPH_TRANSITION_STYLE, transitionStyle.getValue() );
        addExtendedAttribute( document, extendedAttributes, EXT_ATTR_FIRST_LINE_DIRECTION, firstLineDirection.getValue() );

        for ( XpdlLinePathBuilder linePath : linePaths )
        {
            addExtendedAttribute( document, extendedAttributes, EXT_ATTR_LINE_PATH_DEF, linePath.build() );
        }

        transition.appendChild( extendedAttributes );

        return transition;
    }
} 