package com.suncode.pwfl.xpdl.builder;

import lombok.Getter;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.UUID;

@Getter
class XpdlProcessBuilder
    extends XpdlBaseBuilder
{

    private final String id;

    private final String name;

    private LocalDateTime created = LocalDateTime.now();

    private String description;

    private final List<XpdlParticipantBuilder> participants = new ArrayList<>();

    private final List<XpdlVariableBuilder> variables = new ArrayList<>();

    private final List<XpdlActivityBuilder> activities = new ArrayList<>();

    private final List<XpdlTransitionBuilder> transitions = new ArrayList<>();

    private final List<XpdlWorkflowPointBuilder> workflowStarts = new ArrayList<>();

    private final List<XpdlWorkflowPointBuilder> workflowEnds = new ArrayList<>();

    private String attachmentDirectory = UUID.randomUUID().toString();

    private XpdlProcessBuilder( String id, String name )
    {
        this.id = id;
        this.name = name;
        this.description = name;
    }

    public static XpdlProcessBuilder create( String id, String name )
    {
        return new XpdlProcessBuilder( id, name );
    }

    public XpdlProcessBuilder withCreated( LocalDateTime created )
    {
        this.created = created;
        return this;
    }

    public XpdlProcessBuilder withCreated( String created )
    {
        this.created = LocalDateTime.parse( created, DATE_TIME_FORMATTER );
        return this;
    }

    public XpdlProcessBuilder withDescription( String description )
    {
        this.description = description;
        return this;
    }

    public XpdlProcessBuilder addVariable( XpdlVariableBuilder variable )
    {
        variables.add( variable );
        return this;
    }

    public XpdlProcessBuilder addParticipant( XpdlParticipantBuilder participant )
    {
        participants.add( participant );
        return this;
    }

    public XpdlProcessBuilder addActivity( XpdlActivityBuilder activity )
    {
        activities.add( activity );
        return this;
    }

    public XpdlProcessBuilder addTransition( XpdlTransitionBuilder transition )
    {
        transitions.add( transition );
        return this;
    }

    public XpdlProcessBuilder addWorkflowStart( XpdlWorkflowPointBuilder workflowStart )
    {
        workflowStarts.add( workflowStart );
        return this;
    }

    public XpdlProcessBuilder addWorkflowEnd( XpdlWorkflowPointBuilder workflowEnd )
    {
        workflowEnds.add( workflowEnd );
        return this;
    }

    public XpdlProcessBuilder withAttachmentDirectory( String attachmentDirectory )
    {
        this.attachmentDirectory = attachmentDirectory;
        return this;
    }

    public List<XpdlParticipantBuilder> getParticipants()
    {
        return new ArrayList<>( participants );
    }

    public List<XpdlActivityBuilder> getActivities()
    {
        return new ArrayList<>( activities );
    }

    @Override
    public Element buildElement( Document document )
    {
        Element workflowProcess = document.createElementNS( XPDL_NAMESPACE, ELEMENT_WORKFLOW_PROCESS );
        workflowProcess.setAttribute( ATTR_ID, id );
        workflowProcess.setAttribute( ATTR_NAME, name );

        Element processHeader = document.createElementNS( XPDL_NAMESPACE, ELEMENT_PROCESS_HEADER );
        Element processCreatedElement = document.createElementNS( XPDL_NAMESPACE, ELEMENT_CREATED );
        processCreatedElement.setTextContent( formatDateTime( created ) );

        Element descriptionElement = document.createElementNS( XPDL_NAMESPACE, ELEMENT_DESCRIPTION );
        descriptionElement.setTextContent( description );

        processHeader.appendChild( processCreatedElement );
        processHeader.appendChild( descriptionElement );
        workflowProcess.appendChild( processHeader );

        workflowProcess.appendChild( buildFormalParameters( document ) );

        Element dataFields = document.createElementNS( XPDL_NAMESPACE, ELEMENT_DATA_FIELDS );

        ArrayList<XpdlVariableBuilder> variablesSorted = new ArrayList<>( variables );
        variablesSorted.sort( Comparator.comparing( XpdlVariableBuilder::getId ) );

        for ( XpdlVariableBuilder variable : variablesSorted )
        {
            dataFields.appendChild( variable.buildElement( document ) );
        }
        workflowProcess.appendChild( dataFields );

        Element participantsElement = document.createElementNS( XPDL_NAMESPACE, ELEMENT_PARTICIPANTS );
        for ( XpdlParticipantBuilder participant : participants )
        {
            participantsElement.appendChild( participant.buildElement( document ) );
        }
        workflowProcess.appendChild( participantsElement );

        Element activitiesElement = document.createElementNS( XPDL_NAMESPACE, ELEMENT_ACTIVITIES );
        for ( XpdlActivityBuilder activity : activities )
        {
            activitiesElement.appendChild( activity.buildElement( document ) );
        }
        workflowProcess.appendChild( activitiesElement );

        Element transitionsElement = document.createElementNS( XPDL_NAMESPACE, ELEMENT_TRANSITIONS );
        for ( XpdlTransitionBuilder transition : transitions )
        {
            transitionsElement.appendChild( transition.buildElement( document ) );
        }
        workflowProcess.appendChild( transitionsElement );

        workflowProcess.appendChild( buildProcessExtendedAttributes( document ) );

        return workflowProcess;
    }

    private Element buildFormalParameters( Document document )
    {
        Element formalParameters = document.createElementNS( XPDL_NAMESPACE, ELEMENT_FORMAL_PARAMETERS );

        Element initiator = document.createElementNS( XPDL_NAMESPACE, ELEMENT_FORMAL_PARAMETER );
        initiator.setAttribute( ATTR_ID, "Initiator" );
        initiator.setAttribute( ATTR_MODE, STRING_MODE_IN );

        Element initiatorDataType = document.createElementNS( XPDL_NAMESPACE, ELEMENT_DATA_TYPE );
        Element initiatorBasicType = document.createElementNS( XPDL_NAMESPACE, ELEMENT_BASIC_TYPE );
        initiatorBasicType.setAttribute( ATTR_TYPE, STRING_TYPE_STRING );
        initiatorDataType.appendChild( initiatorBasicType );

        Element initiatorDescription = document.createElementNS( XPDL_NAMESPACE, ELEMENT_DESCRIPTION );
        initiatorDescription.setTextContent( "Inicjator procesu" );
        initiator.appendChild( initiatorDataType );
        initiator.appendChild( initiatorDescription );
        formalParameters.appendChild( initiator );

        Element procId = document.createElementNS( XPDL_NAMESPACE, ELEMENT_FORMAL_PARAMETER );
        procId.setAttribute( ATTR_ID, "ProccessId" );
        procId.setAttribute( ATTR_MODE, STRING_MODE_IN );

        Element procIdDataType = document.createElementNS( XPDL_NAMESPACE, ELEMENT_DATA_TYPE );
        Element procIdBasicType = document.createElementNS( XPDL_NAMESPACE, ELEMENT_BASIC_TYPE );
        procIdBasicType.setAttribute( ATTR_TYPE, STRING_TYPE_STRING );
        procIdDataType.appendChild( procIdBasicType );

        Element procIdDescription = document.createElementNS( XPDL_NAMESPACE, ELEMENT_DESCRIPTION );
        procIdDescription.setTextContent( "Identyfikator procesu" );
        procId.appendChild( procIdDataType );
        procId.appendChild( procIdDescription );

        formalParameters.appendChild( procId );

        Element activityId = document.createElementNS( XPDL_NAMESPACE, ELEMENT_FORMAL_PARAMETER );
        activityId.setAttribute( ATTR_ID, "ActivityId" );
        activityId.setAttribute( ATTR_MODE, STRING_MODE_IN );

        Element activityIdDataType = document.createElementNS( XPDL_NAMESPACE, ELEMENT_DATA_TYPE );
        Element activityIdBasicType = document.createElementNS( XPDL_NAMESPACE, ELEMENT_BASIC_TYPE );
        activityIdBasicType.setAttribute( ATTR_TYPE, STRING_TYPE_STRING );
        activityIdDataType.appendChild( activityIdBasicType );

        Element activityIdDescription = document.createElementNS( XPDL_NAMESPACE, ELEMENT_DESCRIPTION );
        activityIdDescription.setTextContent( "Identyfikator zadania" );
        activityId.appendChild( activityIdDataType );
        activityId.appendChild( activityIdDescription );

        formalParameters.appendChild( activityId );

        Element transaction = document.createElementNS( XPDL_NAMESPACE, ELEMENT_FORMAL_PARAMETER );
        transaction.setAttribute( ATTR_ID, "Transaction" );
        transaction.setAttribute( ATTR_MODE, STRING_MODE_IN );

        Element transactionDataType = document.createElementNS( XPDL_NAMESPACE, ELEMENT_DATA_TYPE );
        Element transactionBasicType = document.createElementNS( XPDL_NAMESPACE, ELEMENT_BASIC_TYPE );
        transactionBasicType.setAttribute( ATTR_TYPE, STRING_TYPE_STRING );
        transactionDataType.appendChild( transactionBasicType );

        Element transactionDescription = document.createElementNS( XPDL_NAMESPACE, ELEMENT_DESCRIPTION );
        transactionDescription.setTextContent( "Transakcja silnika Shark" );
        transaction.appendChild( transactionDataType );
        transaction.appendChild( transactionDescription );

        formalParameters.appendChild( transaction );

        Element action = document.createElementNS( XPDL_NAMESPACE, ELEMENT_FORMAL_PARAMETER );
        action.setAttribute( ATTR_ID, "Action" );
        action.setAttribute( ATTR_MODE, STRING_MODE_INOUT );

        Element actionDataType = document.createElementNS( XPDL_NAMESPACE, ELEMENT_DATA_TYPE );
        Element actionBasicType = document.createElementNS( XPDL_NAMESPACE, ELEMENT_BASIC_TYPE );
        actionBasicType.setAttribute( ATTR_TYPE, STRING_TYPE_STRING );
        actionDataType.appendChild( actionBasicType );

        Element actionDescription = document.createElementNS( XPDL_NAMESPACE, ELEMENT_DESCRIPTION );
        actionDescription.setTextContent( "Opisuje przekierowanie do zadań" );
        action.appendChild( actionDataType );
        action.appendChild( actionDescription );

        formalParameters.appendChild( action );

        Element storageData = document.createElementNS( XPDL_NAMESPACE, ELEMENT_FORMAL_PARAMETER );
        storageData.setAttribute( ATTR_ID, "StorageData" );
        storageData.setAttribute( ATTR_MODE, STRING_MODE_INOUT );

        Element storageDataDataType = document.createElementNS( XPDL_NAMESPACE, ELEMENT_DATA_TYPE );
        Element storageDataBasicType = document.createElementNS( XPDL_NAMESPACE, ELEMENT_BASIC_TYPE );
        storageDataBasicType.setAttribute( ATTR_TYPE, STRING_TYPE_STRING );
        storageDataDataType.appendChild( storageDataBasicType );

        Element storageDataDescription = document.createElementNS( XPDL_NAMESPACE, ELEMENT_DESCRIPTION );
        storageDataDescription.setTextContent( "Pełni funkcję schowka danych w ramach procesu" );
        storageData.appendChild( storageDataDataType );
        storageData.appendChild( storageDataDescription );

        formalParameters.appendChild( storageData );

        return formalParameters;
    }

    private Element buildProcessExtendedAttributes( Document document )
    {
        Element extendedAttributes = document.createElementNS( XPDL_NAMESPACE, ELEMENT_EXTENDED_ATTRIBUTES );

        StringBuilder participantOrder = new StringBuilder();
        for ( int i = 0; i < participants.size(); i++ )
        {
            if ( i > 0 )
            {
                participantOrder.append( "," );
            }
            participantOrder.append( participants.get( i ).getId() );
        }
        addExtendedAttribute( document, extendedAttributes, EXT_ATTR_JAWE_GRAPH_WORKFLOW_PARTICIPANT_ORDER, participantOrder.toString() );

        for ( XpdlWorkflowPointBuilder workflowStartBuilder : workflowStarts )
        {
            addExtendedAttribute(
                document, extendedAttributes, EXT_ATTR_JAWE_GRAPH_START_OF_WORKFLOW,
                workflowStartBuilder.toString()
            );
        }

        for ( XpdlWorkflowPointBuilder workflowEndBuilder : workflowEnds )
        {
            addExtendedAttribute(
                document, extendedAttributes, EXT_ATTR_JAWE_GRAPH_END_OF_WORKFLOW,
                workflowEndBuilder.toString()
            );
        }

        addExtendedAttribute( document, extendedAttributes, EXT_ATTR_PROCESS_NAME_MASK, name + "_ VER/#####" );
        addExtendedAttribute( document, extendedAttributes, EXT_ATTR_TABLES, "[]" );
        addExtendedAttribute( document, extendedAttributes, EXT_ATTR_ATTACHMENT_DIRECTORY, attachmentDirectory );

        return extendedAttributes;
    }
} 