package com.suncode.pwfl.xpdl.builder;

import com.suncode.pwfl.workflow.process.map.Activity;
import com.suncode.pwfl.workflow.process.map.Package;
import com.suncode.pwfl.workflow.process.map.Process;
import com.suncode.pwfl.workflow.process.map.Transition;
import com.suncode.pwfl.workflow.process.map.Variable;
import com.suncode.pwfl.workflow.process.map.VariableRef;
import com.suncode.pwfl.workflow.process.map.VariableType;
import lombok.Setter;

import java.time.LocalDateTime;
import java.util.Comparator;
import java.util.List;
import java.util.UUID;
import java.util.function.Supplier;

public class XpdlBuilderService
{

    private static final int ACTIVITY_BLOCK_WIDTH = 90;

    private static final int ACTIVITY_BLOCK_HEIGHT = 60;

    private static final int ACTIVITY_BLOCK_MARGIN = 50;

    private static final int WORKFLOW_POINT_SIZE = 40;

    private static final int PARTICIPANT_LABEL_OFFSET = 30;

    @Setter
    private static Supplier<String> attachmentDirectoryIdGenerator = () -> UUID.randomUUID().toString();

    public static String buildBy( Package pkg )
    {
        return buildBy( pkg, LocalDateTime.now() );
    }

    static String buildBy( Package pkg, LocalDateTime createdTimestamp )
    {
        XpdlBuilder xpdlBuilder = new XpdlBuilder()
            .withPackageId( pkg.getId() )
            .withPackageName( pkg.getName() )
            .withAuthor( "Suncode" )
            .withCreated( createdTimestamp );

        for ( Process process : pkg.getProcesses() )
        {
            appendProcess( createdTimestamp, process, xpdlBuilder );
        }

        return xpdlBuilder.build();
    }

    private static void appendProcess( LocalDateTime createdTimestamp, Process process, XpdlBuilder xpdlBuilder )
    {
        XpdlProcessBuilder processBuilder = XpdlProcessBuilder
            .create( process.getId(), process.getName() )
            .withCreated( createdTimestamp )
            .withAttachmentDirectory( attachmentDirectoryIdGenerator.get() )
            .addParticipant(
                XpdlParticipantBuilder.create( "uczestnik_1", "Uczestnik 1" )
            );
        xpdlBuilder.addProcess( processBuilder );

        for ( Variable variable : process.getVariables() )
        {
            VariableType type = variable.getType();
            XpdlVariableBuilder variableBuilder = switch ( type )
            {
                case STRING -> XpdlVariableBuilder.createString( variable.getId(), variable.getName() );
                case INTEGER -> XpdlVariableBuilder.createInteger( variable.getId(), variable.getName() );
                case FLOAT -> XpdlVariableBuilder.createFloat( variable.getId(), variable.getName() );
                case BOOLEAN -> XpdlVariableBuilder.createBoolean( variable.getId(), variable.getName() );
                case DATE -> XpdlVariableBuilder.createDate( variable.getId(), variable.getName() );
                case DATE_TIME -> XpdlVariableBuilder.createDateTime( variable.getId(), variable.getName() );
            };

            variableBuilder = variableBuilder
                .withCreationDate( createdTimestamp )
                .withModificationDate( createdTimestamp );

            processBuilder.addVariable( variableBuilder );
        }

        int processStartX = PARTICIPANT_LABEL_OFFSET + ACTIVITY_BLOCK_MARGIN;
        int processStartY = ACTIVITY_BLOCK_MARGIN;

        int processEndX = processStartX + WORKFLOW_POINT_SIZE + ACTIVITY_BLOCK_MARGIN;
        int processEndY = processStartY;

        for ( Activity activity : process.getActivities() )
        {
            List<XpdlActivityVariableBuilder> activityVariables = activity.getVariableRefs().stream()
                .sorted( Comparator.comparing( VariableRef::getPosition ) )
                .map( variableRef -> new XpdlActivityVariableBuilder( variableRef.getId(), variableRef.isEditable() ) )
                .toList();

            XpdlActivityBuilder activityBuilder = XpdlActivityBuilder
                .create( activity.getId(), activity.getName(), "uczestnik_1", processEndX, processEndY )
                .withVariables( activityVariables );

            processBuilder.addActivity( activityBuilder );

            processEndX += ACTIVITY_BLOCK_WIDTH + ACTIVITY_BLOCK_MARGIN;
        }

        for ( Transition transition : process.getTransitions() )
        {
            XpdlActivityBuilder fromActivityBuilder = processBuilder.getActivities().stream()
                .filter( activity -> activity.getId().equals( transition.getFromActivityId() ) )
                .findFirst()
                .orElseThrow();

            XpdlActivityBuilder toActivityBuilder = processBuilder.getActivities().stream()
                .filter( activity -> activity.getId().equals( transition.getToActivityId() ) )
                .findFirst()
                .orElseThrow();

            XpdlTransitionBuilder transitionBuilder = XpdlTransitionBuilder
                .create( transition.getId(), transition.getFromActivityId(), transition.getToActivityId() )
                .addLinePath(
                    XpdlLinePathBuilder.create()
                        .moveTo( fromActivityBuilder.getXOffset() + ACTIVITY_BLOCK_WIDTH,
                                 fromActivityBuilder.getYOffset() + (ACTIVITY_BLOCK_HEIGHT / 2) )
                        .lineTo( toActivityBuilder.getXOffset(),
                                 toActivityBuilder.getYOffset() + (ACTIVITY_BLOCK_HEIGHT / 2) )
                        .closePath()
                )
                .addLinePath(
                    XpdlLinePathBuilder.create()
                        .rightArrow( toActivityBuilder.getXOffset(),
                                     toActivityBuilder.getYOffset() + (ACTIVITY_BLOCK_HEIGHT / 2) )
                );

            processBuilder.addTransition( transitionBuilder );
        }

        Activity firstActivity = process.getActivities().get( 0 );
        Activity lastActivity = process.getActivities().get( process.getActivities().size() - 1 );

        processBuilder
            .addWorkflowStart( XpdlWorkflowPointBuilder.create(
                "uczestnik_1", firstActivity.getId(),
                processStartX,
                processStartY + ((ACTIVITY_BLOCK_HEIGHT - WORKFLOW_POINT_SIZE) / 2)
            ) )
            .addWorkflowEnd( XpdlWorkflowPointBuilder.create(
                "uczestnik_1", lastActivity.getId(),
                processEndX,
                processEndY + ((ACTIVITY_BLOCK_HEIGHT - WORKFLOW_POINT_SIZE) / 2)
            ) );
    }

}
