/*
 * Decompiled with CFR 0.152.
 */
package com.codeborne.selenide.impl;

import com.codeborne.selenide.Browser;
import com.codeborne.selenide.Config;
import com.codeborne.selenide.DownloadsFolder;
import com.codeborne.selenide.Driver;
import com.codeborne.selenide.ex.FileNotDownloadedError;
import com.codeborne.selenide.files.DownloadAction;
import com.codeborne.selenide.files.DownloadedFile;
import com.codeborne.selenide.files.FileFilter;
import com.codeborne.selenide.impl.Downloader;
import com.codeborne.selenide.impl.Downloads;
import com.codeborne.selenide.impl.FileHelper;
import com.codeborne.selenide.impl.WebElementSource;
import java.io.File;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.jspecify.annotations.Nullable;
import org.openqa.selenium.WebElement;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DownloadFileToFolder {
    private static final Logger log = LoggerFactory.getLogger(DownloadFileToFolder.class);
    private static final Set<String> CHROMIUM_TEMPORARY_FILES = Set.of("crdownload", "tmp");
    private static final Set<String> FIREFOX_TEMPORARY_FILES = Set.of("part");
    private final Downloader downloader;

    DownloadFileToFolder(Downloader downloader) {
        this.downloader = downloader;
    }

    public DownloadFileToFolder() {
        this(new Downloader());
    }

    public File download(WebElementSource link, WebElement clickable, long timeout, long incrementTimeout, FileFilter fileFilter, DownloadAction action) {
        long minimalIncrementTimeout = Math.max(incrementTimeout, 1000L);
        return this.clickAndWaitForNewFilesInDownloadsFolder(link, clickable, timeout, minimalIncrementTimeout, fileFilter, action);
    }

    File clickAndWaitForNewFilesInDownloadsFolder(WebElementSource link, WebElement clickable, long timeout, long incrementTimeout, FileFilter fileFilter, DownloadAction action) {
        Driver driver = link.driver();
        Config config = driver.config();
        long pollingInterval = Math.max(config.pollingInterval(), 50L);
        DownloadsFolder folder = this.getDownloadsFolder(driver);
        if (folder == null) {
            throw new IllegalStateException("Downloads folder is not configured");
        }
        folder.cleanupBeforeDownload();
        long downloadStartedAt = System.currentTimeMillis();
        action.perform(driver, clickable);
        this.waitForNewFiles(driver, fileFilter, folder, downloadStartedAt, timeout, incrementTimeout, pollingInterval);
        this.waitUntilDownloadsCompleted(driver, folder, fileFilter, timeout, incrementTimeout, pollingInterval);
        Downloads newDownloads = new Downloads(folder.filesNewerThan(downloadStartedAt));
        if (log.isInfoEnabled()) {
            log.info("Downloaded files in {}: {}", (Object)folder, (Object)newDownloads.filesAsString());
        }
        if (log.isDebugEnabled()) {
            log.debug("All downloaded files: {}", (Object)folder.filesAsString());
        }
        File downloadedFile = newDownloads.firstDownloadedFile(timeout, fileFilter);
        return this.archiveFile(driver, downloadedFile);
    }

    protected @Nullable DownloadsFolder getDownloadsFolder(Driver driver) {
        return driver.browserDownloadsFolder();
    }

    void waitUntilDownloadsCompleted(Driver driver, DownloadsFolder folder, FileFilter filter, long timeout, long incrementTimeout, long pollingInterval) {
        Browser browser = driver.browser();
        if (browser.isChrome() || browser.isEdge()) {
            this.waitUntilFileDisappears(driver, folder, CHROMIUM_TEMPORARY_FILES, filter, timeout, incrementTimeout, pollingInterval);
        } else if (browser.isFirefox()) {
            this.waitUntilFileDisappears(driver, folder, FIREFOX_TEMPORARY_FILES, filter, timeout, incrementTimeout, pollingInterval);
        } else {
            this.waitWhileFilesAreBeingModified(driver, folder, timeout, pollingInterval);
        }
    }

    private void waitUntilFileDisappears(Driver driver, DownloadsFolder folder, Set<String> extension, FileFilter filter, long timeout, long incrementTimeout, long pollingInterval) {
        long start = System.currentTimeMillis();
        while (System.currentTimeMillis() - start <= timeout) {
            if (!folder.hasFiles(extension, filter)) {
                log.debug("No {} files found, conclude download is completed (filter: {})", extension, (Object)filter);
                return;
            }
            log.debug("Found {} files, waiting for {} ms (filter: {})...", new Object[]{extension, pollingInterval, filter});
            this.failFastIfNoChanges(driver, folder, filter, start, timeout, incrementTimeout);
            this.pause(pollingInterval);
        }
        if (folder.hasFiles(extension, filter)) {
            String message = String.format("Folder %s still contains files %s after %s ms. Apparently, the downloading hasn't completed in time.", folder, extension, timeout);
            throw new FileNotDownloadedError(message, timeout);
        }
    }

    protected void waitWhileFilesAreBeingModified(Driver driver, DownloadsFolder folder, long timeout, long pollingInterval) {
        Map<String, Long> times = folder.modificationTimes();
        long lastModifiedAt = System.currentTimeMillis();
        long start = System.currentTimeMillis();
        while (System.currentTimeMillis() - start < timeout) {
            Map<String, Long> newTimes = folder.modificationTimes();
            if (!newTimes.equals(times)) {
                log.debug("Files has been modified - old: {}, new: {}", times, newTimes);
                lastModifiedAt = System.currentTimeMillis();
                times = newTimes;
            } else {
                log.debug("Files has not been modified in last {} ms: {}", (Object)pollingInterval, times);
                if (System.currentTimeMillis() - lastModifiedAt > 1000L) {
                    log.debug("Files has not been modified during last {} ms.", (Object)(System.currentTimeMillis() - lastModifiedAt));
                    return;
                }
            }
            this.pause(pollingInterval);
        }
        log.warn("Files are still being modified during last {} ms.", (Object)(System.currentTimeMillis() - lastModifiedAt));
    }

    void waitForNewFiles(Driver driver, FileFilter fileFilter, DownloadsFolder folder, long clickMoment, long timeout, long incrementTimeout, long pollingInterval) {
        if (log.isDebugEnabled()) {
            log.debug("Waiting for files in {}...", (Object)folder);
        }
        long start = System.currentTimeMillis();
        while (System.currentTimeMillis() - start <= timeout) {
            Downloads downloads = new Downloads(folder.filesNewerThan(clickMoment));
            List<DownloadedFile> matchingFiles = downloads.files(fileFilter);
            if (!matchingFiles.isEmpty()) {
                log.debug("Matching files found: {}, all new files: {}, all files: {}", new Object[]{matchingFiles, downloads.filesAsString(), folder.filesAsString()});
                return;
            }
            log.debug("Matching files not found: {}, all new files: {}, all files: {}", new Object[]{matchingFiles, downloads.filesAsString(), folder.filesAsString()});
            this.failFastIfNoChanges(driver, folder, fileFilter, start, timeout, incrementTimeout);
            this.pause(pollingInterval);
        }
        log.debug("Matching files still not found -> stop waiting for new files after {} ms. (timeout: {} ms.)", (Object)(System.currentTimeMillis() - start), (Object)timeout);
    }

    protected void failFastIfNoChanges(Driver driver, DownloadsFolder folder, FileFilter filter, long start, long timeout, long incrementTimeout) {
        long lastFileUpdate = folder.lastModificationTime().orElse(start);
        long now = System.currentTimeMillis();
        long filesHasNotBeenUpdatedForMs = this.filesHasNotBeenUpdatedForMs(start, now, lastFileUpdate);
        if (filesHasNotBeenUpdatedForMs > incrementTimeout) {
            String message = String.format("Failed to download file%s in %d ms: files in %s haven't been modified for %s ms. (started at: %s, lastFileUpdate: %s, now: %s, incrementTimeout: %s)%nModification times: %s", filter.description(), timeout, folder, filesHasNotBeenUpdatedForMs, start, lastFileUpdate, now, incrementTimeout, folder.modificationTimes());
            throw new FileNotDownloadedError(message, timeout);
        }
    }

    long filesHasNotBeenUpdatedForMs(long downloadStartedAt, long now, long lastFileUpdate) {
        return now - Math.max(lastFileUpdate, downloadStartedAt);
    }

    private void pause(long milliseconds) {
        try {
            Thread.sleep(milliseconds);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException(e);
        }
    }

    protected File archiveFile(Driver driver, File downloadedFile) {
        File uniqueFolder = this.downloader.prepareTargetFolder(driver.config());
        File archivedFile = new File(uniqueFolder, downloadedFile.getName());
        FileHelper.moveFile(downloadedFile, archivedFile);
        log.debug("Moved the downloaded file {} to {}", (Object)downloadedFile, (Object)archivedFile);
        return archivedFile;
    }
}

